import type { Metadata } from "next"
import BlogArticleLayout from "@/components/blog-article-layout"
import { getRelatedArticles, getPreviousArticle, getNextArticle } from "@/lib/blog-utils"

export const metadata: Metadata = {
  title: "DPE 2024 : Ce Qui Change pour les Propriétaires Bailleurs | Doges",
  description:
    "Découvrez les changements du DPE en 2024, les nouvelles obligations pour les propriétaires bailleurs et comment vous adapter à cette réforme énergétique.",
  openGraph: {
    title: "DPE 2024 : Ce Qui Change pour les Propriétaires Bailleurs | Doges",
    description:
      "Découvrez les changements du DPE en 2024, les nouvelles obligations pour les propriétaires bailleurs et comment vous adapter à cette réforme énergétique.",
    images: [{ url: "/dpe-energy-rating.png", width: 1200, height: 630 }],
  },
}

const tableOfContents = [
  { id: "introduction", text: "Introduction", level: 2 },
  { id: "rappel-dpe", text: "Rappel : qu'est-ce que le DPE ?", level: 2 },
  { id: "evolution-dpe", text: "L'évolution du DPE depuis 2021", level: 2 },
  { id: "changements-2024", text: "Les changements du DPE en 2024", level: 2 },
  { id: "nouvelle-methode", text: "La nouvelle méthode de calcul", level: 3 },
  { id: "fiabilite-renforcee", text: "Une fiabilité renforcée", level: 3 },
  { id: "validite-dpe", text: "Durée de validité du DPE", level: 3 },
  { id: "impact-proprietaires", text: "Impact pour les propriétaires bailleurs", level: 2 },
  { id: "interdiction-location", text: "Calendrier d'interdiction de location", level: 3 },
  { id: "gel-loyers", text: "Gel des loyers pour les passoires thermiques", level: 3 },
  { id: "obligation-travaux", text: "Obligation de travaux en copropriété", level: 3 },
  { id: "audit-energetique", text: "L'audit énergétique obligatoire", level: 2 },
  { id: "contenu-audit", text: "Contenu de l'audit énergétique", level: 3 },
  { id: "cout-audit", text: "Coût et réalisation de l'audit", level: 3 },
  { id: "strategies-adaptation", text: "Stratégies d'adaptation pour les propriétaires", level: 2 },
  { id: "anticiper-dpe", text: "Anticiper la réalisation d'un DPE", level: 3 },
  { id: "planifier-travaux", text: "Planifier les travaux de rénovation", level: 3 },
  { id: "aides-financieres", text: "Mobiliser les aides financières", level: 3 },
  { id: "conclusion", text: "Conclusion", level: 2 },
  { id: "faq", text: "Questions fréquentes", level: 2 },
]

const author = {
  name: "Bedis Iddir",
  role: "Expert en gestion locative",
  image: "/experts/bedis-iddir.jpg",
  bio: "Spécialiste en gestion locative avec plus de 10 ans d'expérience dans l'immobilier parisien. Expert en réglementation et optimisation de patrimoine immobilier.",
  linkedin: "https://www.linkedin.com/in/bedis-iddir/",
}

export default function DPE2024() {
  const slug = "dpe-2024-changements-proprietaires"
  const category = "Réglementation"
  const relatedArticles = getRelatedArticles(slug, category, 2)
  const previousArticle = getPreviousArticle(slug)
  const nextArticle = getNextArticle(slug)

  return (
    <BlogArticleLayout
      title="DPE 2024 : Ce Qui Change pour les Propriétaires Bailleurs"
      date="10 mars 2025"
      category={category}
      readingTime="8 min"
      image="/dpe-energy-rating.png"
      tableOfContents={tableOfContents}
      author={author}
      previousArticle={previousArticle}
      nextArticle={nextArticle}
      relatedArticles={relatedArticles}
    >
      <h2 id="introduction">Introduction</h2>
      <p>
        Le Diagnostic de Performance Énergétique (DPE) est devenu un élément central de la politique de transition
        énergétique en France. En 2024, de nouvelles évolutions viennent renforcer son importance et son impact sur le
        marché immobilier, particulièrement pour les propriétaires bailleurs. Ces changements s'inscrivent dans la
        continuité de la réforme majeure de 2021 et visent à accélérer la rénovation du parc immobilier français, dont
        près de 17% des logements sont considérés comme des passoires thermiques.
      </p>
      <p>
        Pour les propriétaires bailleurs, ces évolutions ne sont pas anodines : elles conditionnent désormais la
        possibilité même de louer un bien, influencent sa valeur sur le marché et peuvent imposer des travaux coûteux.
        Cet article fait le point sur les changements du DPE en 2024, leurs implications concrètes pour les
        propriétaires et les stratégies à adopter pour s'y adapter efficacement.
      </p>

      <h2 id="rappel-dpe">Rappel : qu'est-ce que le DPE ?</h2>
      <p>
        Le Diagnostic de Performance Énergétique est un document qui évalue la consommation d'énergie d'un logement et
        son impact en termes d'émissions de gaz à effet de serre. Il classe les biens immobiliers sur deux échelles :
      </p>
      <ul>
        <li>Une échelle de classe énergétique, de A (très performant) à G (très énergivore)</li>
        <li>Une échelle d'émission de gaz à effet de serre, également de A à G</li>
      </ul>
      <p>
        Le DPE est obligatoire pour toute vente ou location d'un bien immobilier. Il doit être présenté aux potentiels
        acquéreurs ou locataires dès la première visite et être annexé au contrat de vente ou de location.
      </p>

      <div className="relative w-full h-[300px] my-6 rounded-lg overflow-hidden">
        <img
          src="/dpe-classes-comparison.png"
          alt="Comparaison des différentes classes énergétiques du DPE"
          className="object-contain w-full h-full"
        />
      </div>

      <h2 id="evolution-dpe">L'évolution du DPE depuis 2021</h2>
      <p>La réforme du DPE entrée en vigueur le 1er juillet 2021 a marqué un tournant majeur :</p>
      <ul>
        <li>Passage d'un DPE informatif à un DPE opposable juridiquement</li>
        <li>
          Nouvelle méthode de calcul basée uniquement sur les caractéristiques physiques du bâtiment (et non plus sur
          les factures)
        </li>
        <li>Prise en compte du confort d'été (résistance à la chaleur)</li>
        <li>Intégration des émissions de gaz à effet de serre dans le classement final</li>
        <li>Affichage des dépenses théoriques d'énergie</li>
      </ul>
      <p>
        Cette réforme a entraîné un reclassement significatif de nombreux biens, avec une augmentation estimée de 800
        000 logements classés F ou G (passoires thermiques).
      </p>

      <h2 id="changements-2024">Les changements du DPE en 2024</h2>

      <h3 id="nouvelle-methode">La nouvelle méthode de calcul</h3>
      <p>En 2024, la méthode de calcul du DPE continue d'évoluer avec :</p>
      <ul>
        <li>Une meilleure prise en compte des spécificités des logements construits avant 1975</li>
        <li>Un ajustement des coefficients de conversion en énergie primaire</li>
        <li>Une évaluation plus précise de l'impact des systèmes de chauffage et de production d'eau chaude</li>
        <li>Une intégration renforcée des énergies renouvelables dans le calcul</li>
      </ul>
      <p>
        Ces ajustements visent à corriger certaines anomalies constatées depuis 2021, notamment pour les logements
        anciens qui étaient parfois pénalisés de façon excessive.
      </p>

      <h3 id="fiabilite-renforcee">Une fiabilité renforcée</h3>
      <p>Pour améliorer la fiabilité des DPE, plusieurs mesures ont été mises en place :</p>
      <ul>
        <li>Renforcement des contrôles sur les diagnostiqueurs</li>
        <li>Obligation de formation continue pour les professionnels</li>
        <li>Mise en place d'un système de vérification aléatoire des DPE réalisés</li>
        <li>Standardisation accrue des méthodes de relevé et de mesure</li>
      </ul>
      <p>
        Ces mesures répondent aux critiques concernant la variabilité des résultats constatée entre différents
        diagnostiqueurs pour un même bien.
      </p>

      <h3 id="validite-dpe">Durée de validité du DPE</h3>
      <p>
        Depuis le 1er juillet 2021, la durée de validité du DPE est de 10 ans. Toutefois, il est important de noter que
        :
      </p>
      <ul>
        <li>
          Les DPE réalisés entre le 1er janvier 2013 et le 31 décembre 2017 sont valides jusqu'au 31 décembre 2022
        </li>
        <li>Les DPE réalisés entre le 1er janvier 2018 et le 30 juin 2021 sont valides jusqu'au 31 décembre 2024</li>
        <li>Les DPE réalisés à partir du 1er juillet 2021 sont valides 10 ans</li>
      </ul>
      <p>
        En 2024, de nombreux propriétaires doivent donc renouveler leur DPE, ce qui peut entraîner un changement de
        classe énergétique avec les nouvelles méthodes de calcul.
      </p>

      <h2 id="impact-proprietaires">Impact pour les propriétaires bailleurs</h2>

      <h3 id="interdiction-location">Calendrier d'interdiction de location</h3>
      <p>
        La loi Climat et Résilience a instauré un calendrier progressif d'interdiction de location des logements
        énergivores :
      </p>
      <div className="bg-gray-100 p-4 rounded-lg my-4">
        <ul className="list-disc pl-5">
          <li>
            <strong>1er janvier 2023</strong> : Gel des loyers pour les logements classés F et G
          </li>
          <li>
            <strong>1er janvier 2025</strong> : Interdiction de louer les logements classés G (consommation supérieure à
            450 kWh/m²/an)
          </li>
          <li>
            <strong>1er janvier 2028</strong> : Interdiction de louer les logements classés F
          </li>
          <li>
            <strong>1er janvier 2034</strong> : Interdiction de louer les logements classés E
          </li>
        </ul>
      </div>
      <p>
        Ces échéances constituent un véritable couperet pour les propriétaires de passoires thermiques, qui doivent
        impérativement engager des travaux de rénovation énergétique s'ils souhaitent continuer à louer leur bien.
      </p>

      <h3 id="gel-loyers">Gel des loyers pour les passoires thermiques</h3>
      <p>
        Depuis le 24 août 2022, les propriétaires de logements classés F ou G ne peuvent plus augmenter le loyer entre
        deux locataires ou lors du renouvellement du bail. Cette mesure, qui s'applique à l'ensemble du territoire
        français, vise à inciter les propriétaires à réaliser des travaux de rénovation énergétique.
      </p>
      <p>
        En 2024, ce gel des loyers continue de s'appliquer et pourrait même être étendu aux logements classés E dans
        certaines zones tendues, selon les discussions législatives en cours.
      </p>

      <h3 id="obligation-travaux">Obligation de travaux en copropriété</h3>
      <p>
        Pour les copropriétés, la loi Climat et Résilience impose la réalisation d'un diagnostic de performance
        énergétique collectif (DPE collectif) et la mise en place d'un plan pluriannuel de travaux (PPT) :
      </p>
      <ul>
        <li>Depuis le 1er janvier 2023 : copropriétés de plus de 200 lots</li>
        <li>Depuis le 1er janvier 2024 : copropriétés de 51 à 200 lots</li>
        <li>À partir du 1er janvier 2025 : copropriétés de moins de 50 lots</li>
      </ul>
      <p>
        Ce plan pluriannuel doit prévoir les travaux nécessaires à la conservation de l'immeuble et à l'amélioration de
        sa performance énergétique sur une période de 10 ans.
      </p>

      <h2 id="audit-energetique">L'audit énergétique obligatoire</h2>

      <h3 id="contenu-audit">Contenu de l'audit énergétique</h3>
      <p>
        Depuis le 1er avril 2023, la vente d'un logement classé F ou G doit être accompagnée d'un audit énergétique. Cet
        audit va plus loin que le DPE en proposant :
      </p>
      <ul>
        <li>Un état des lieux complet du logement</li>
        <li>Des scénarios de travaux pour atteindre une classe énergétique plus performante</li>
        <li>Une estimation du coût des travaux et des économies d'énergie générées</li>
        <li>Des informations sur les aides financières disponibles</li>
      </ul>
      <p>
        À partir du 1er janvier 2025, cette obligation sera étendue aux logements classés E, puis aux logements classés
        D à partir du 1er janvier 2034.
      </p>

      <h3 id="cout-audit">Coût et réalisation de l'audit</h3>
      <p>
        Le coût d'un audit énergétique varie généralement entre 400€ et 800€, selon la taille et la complexité du
        logement. Il doit être réalisé par un professionnel certifié, indépendant et impartial.
      </p>
      <p>
        Cet audit constitue un investissement non négligeable, mais il permet d'obtenir une feuille de route précise
        pour les travaux de rénovation et peut constituer un argument de vente pour les acquéreurs sensibles aux
        questions énergétiques.
      </p>

      <h2 id="strategies-adaptation">Stratégies d'adaptation pour les propriétaires</h2>

      <h3 id="anticiper-dpe">Anticiper la réalisation d'un DPE</h3>
      <p>Pour les propriétaires bailleurs, il est recommandé de :</p>
      <ul>
        <li>Vérifier la date de validité de l'ancien DPE</li>
        <li>Réaliser un nouveau DPE avant la mise en location ou la vente</li>
        <li>Faire appel à plusieurs diagnostiqueurs pour comparer les approches et les résultats</li>
        <li>Préparer les documents techniques du logement pour faciliter le diagnostic</li>
      </ul>
      <p>
        Cette anticipation permet d'éviter les mauvaises surprises et de disposer du temps nécessaire pour planifier
        d'éventuels travaux.
      </p>

      <h3 id="planifier-travaux">Planifier les travaux de rénovation</h3>
      <p>Face aux échéances d'interdiction de location, une planification stratégique des travaux s'impose :</p>
      <ul>
        <li>Prioriser les interventions à fort impact énergétique (isolation, chauffage)</li>
        <li>Échelonner les travaux sur plusieurs années pour étaler l'investissement</li>
        <li>Profiter des périodes de vacance locative pour réaliser les travaux les plus invasifs</li>
        <li>Coordonner les interventions avec les travaux de la copropriété</li>
      </ul>
      <p>
        Un plan de travaux bien conçu permet d'optimiser les investissements et de maximiser le gain de classes
        énergétiques.
      </p>

      <h3 id="aides-financieres">Mobiliser les aides financières</h3>
      <p>De nombreuses aides sont disponibles pour financer les travaux de rénovation énergétique :</p>
      <ul>
        <li>MaPrimeRénov' (jusqu'à 35 000€ pour une rénovation globale)</li>
        <li>L'éco-prêt à taux zéro (jusqu'à 50 000€)</li>
        <li>La TVA à taux réduit (5,5%)</li>
        <li>Les Certificats d'Économie d'Énergie (CEE)</li>
        <li>Les aides locales (régions, départements, communes)</li>
        <li>Les dispositifs de défiscalisation (Denormandie, déficit foncier)</li>
      </ul>
      <p>La combinaison de ces aides peut réduire significativement le reste à charge pour les propriétaires.</p>

      <h2 id="conclusion">Conclusion</h2>
      <p>
        Les évolutions du DPE en 2024 renforcent son rôle central dans la politique de transition énergétique du parc
        immobilier français. Pour les propriétaires bailleurs, ces changements représentent à la fois un défi et une
        opportunité :
      </p>
      <ul>
        <li>Un défi, car ils imposent des contraintes réglementaires et financières croissantes</li>
        <li>
          Une opportunité, car la rénovation énergétique valorise le patrimoine et améliore son attractivité locative
        </li>
      </ul>
      <p>
        L'anticipation et la planification stratégique sont les maîtres mots pour s'adapter efficacement à ces
        évolutions. Les propriétaires qui agiront rapidement bénéficieront non seulement des aides financières
        actuellement disponibles, mais aussi d'un avantage concurrentiel sur un marché locatif de plus en plus sensible
        à la performance énergétique.
      </p>
      <p>
        Chez Doges, nous accompagnons les propriétaires bailleurs dans cette transition, en leur proposant un diagnostic
        personnalisé et des solutions adaptées à leur situation. N'hésitez pas à nous contacter pour bénéficier de notre
        expertise en matière de rénovation énergétique et de gestion locative.
      </p>

      <div className="bg-blue-50 border-l-4 border-blue-500 p-4 my-8">
        <p className="font-semibold">Besoin d'accompagnement pour la rénovation énergétique de votre bien ?</p>
        <p className="mt-2">
          Nos experts vous aident à optimiser votre stratégie et à maximiser les aides financières disponibles.
        </p>
        <a
          href="/contact"
          className="inline-block mt-4 px-4 py-2 bg-blue-600 text-white rounded hover:bg-blue-700 transition-colors"
        >
          Prendre rendez-vous
        </a>
      </div>

      <h2 id="faq">Questions fréquentes</h2>

      <div className="space-y-4 mt-4">
        <div className="border-b pb-4">
          <h3 className="font-semibold">Mon DPE a été réalisé en 2019. Est-il encore valable ?</h3>
          <p>
            Les DPE réalisés entre le 1er janvier 2018 et le 30 juin 2021 sont valides jusqu'au 31 décembre 2024. Vous
            devrez donc faire réaliser un nouveau DPE avant cette date.
          </p>
        </div>

        <div className="border-b pb-4">
          <h3 className="font-semibold">Puis-je contester le résultat d'un DPE ?</h3>
          <p>
            Oui, depuis que le DPE est devenu opposable (1er juillet 2021), vous pouvez contester son résultat si vous
            estimez qu'il comporte des erreurs. La contestation doit être adressée au diagnostiqueur dans un premier
            temps, puis éventuellement portée devant les tribunaux.
          </p>
        </div>

        <div className="border-b pb-4">
          <h3 className="font-semibold">
            Quels sont les travaux les plus efficaces pour gagner des classes énergétiques ?
          </h3>
          <p>
            Les travaux ayant généralement le plus d'impact sont l'isolation thermique (toiture, murs, planchers bas),
            le remplacement des fenêtres simple vitrage, la modernisation du système de chauffage et l'installation
            d'une ventilation performante. L'audit énergétique vous permettra d'identifier les interventions
            prioritaires pour votre logement spécifique.
          </p>
        </div>

        <div className="border-b pb-4">
          <h3 className="font-semibold">
            Je suis copropriétaire. Puis-je réaliser des travaux d'amélioration énergétique sans l'accord de la
            copropriété ?
          </h3>
          <p>
            Certains travaux peuvent être réalisés sans l'accord de la copropriété, comme le remplacement des fenêtres
            ou l'installation d'un système de chauffage individuel. En revanche, les travaux affectant les parties
            communes (isolation par l'extérieur, réfection de toiture) nécessitent un vote en assemblée générale.
          </p>
        </div>

        <div>
          <h3 className="font-semibold">Le DPE est-il obligatoire pour tous les types de biens ?</h3>
          <p>
            Le DPE est obligatoire pour tous les logements mis en vente ou en location, à quelques exceptions près : les
            constructions provisoires, les bâtiments indépendants de moins de 50 m², les monuments historiques, certains
            bâtiments agricoles et les logements destinés à être occupés moins de 4 mois par an.
          </p>
        </div>
      </div>
    </BlogArticleLayout>
  )
}
