"use client"

import type React from "react"

import {useState, useEffect, useMemo} from "react"
import {
    ChevronRight,
    Building2,
    ArrowRight,
    MapPin,
    Clock,
    Quote,
    Calendar,
    Users,
    Shield,
    Facebook,
    Twitter,
    Linkedin,
    Instagram,
    MessageCircle,
} from "lucide-react"
import Image from "next/image"
import Link from "next/link"
import {Button} from "@/components/ui/button"
import {motion, useScroll} from "framer-motion"
import {blogArticles} from "@/lib/blog-utils"
import NationalPresence from "@/components/national-presence"
import ServiceCardOptimized from "@/components/service-card-optimized"
// Importer le hook mais ne pas l'utiliser pour l'instant
// import { useServiceImagesPreloader } from "@/hooks/use-image-preloader"

export default function HomePage() {
    const [isScrolled, setIsScrolled] = useState(false)
    const {scrollYProgress} = useScroll()

    // Commentons temporairement le préchargeur d'images pour éviter l'erreur
    // const serviceImagesStatus = useServiceImagesPreloader()

    useEffect(() => {
        const handleScroll = () => {
            setIsScrolled(window.scrollY > 50)
        }
        window.addEventListener("scroll", handleScroll)
        return () => window.removeEventListener("scroll", handleScroll)
    }, [])

    // Utiliser useMemo pour éviter de recréer le tableau à chaque rendu
    const services = useMemo(
        () => [
            {
                title: "Gestion Locative",
                description: "Confiez-nous la gestion de votre bien et profitez d'une tranquillité d'esprit totale.",
                image: "/modern-apartment-interior.png",
                link: "/services/gestion-locative",
            },
            {
                title: "Transaction Immobilière",
                description: "Vendez ou achetez votre bien en toute sérénité grâce à notre expertise du marché.",
                image: "/images/transaction-service.webp",
                link: "/services/transaction",
            },
            {
                title: "Garantie Loyers Impayés",
                description: "Protégez vos revenus locatifs avec notre assurance contre les loyers impayés.",
                image: "/rental-income-protection-security.png",
                link: "/services/garantie-loyers-impayes",
            },
            {
                title: "Assurance PNO",
                description: "Protégez votre bien immobilier avec notre assurance propriétaire non occupant.",
                image: "/images/assurance-pno-service.webp",
                link: "/services/assurance-pno",
            },
            {
                title: "Comptabilité Mandants",
                description: "Bénéficiez d'un suivi comptable rigoureux et de déclarations fiscales optimisées.",
                image: "/real-estate-accounting-documents.png",
                link: "/services/comptabilite-mandants",
            },
            {
                title: "Juridique",
                description: "Bénéficiez de conseils juridiques personnalisés pour toutes vos questions immobilières.",
                image: "/legal-real-estate-professional-contract.png",
                link: "/services/juridique",
            },
            {
                title: "Service Commercial",
                description: "Notre équipe commerciale vous accompagne dans tous vos projets immobiliers.",
                image: "/images/commercial-service.webp",
                link: "/services/service-commercial",
            },
            {
                title: "Professionnels",
                description: "Des solutions adaptées aux besoins spécifiques des professionnels de l'immobilier.",
                image: "/images/professional-services.png",
                link: "/services/professionnels",
            },
        ],
        [],
    )

    const team = [
        {
            name: "Jean Dupont",
            role: "Directeur Général",
            description: "Plus de 20 ans d'expérience dans l'administration de biens.",
            image:
                "https://images.unsplash.com/photo-1500648767791-00d5a4ee9aa5?ixlib=rb-=eyJhcHBfaWQiOjEyMDd9&auto=format&fit=facearea&facepad=8&w=256&h=256&q=80",
        },
        {
            name: "Sophie Martin",
            role: "Responsable Location",
            description: "Spécialiste de la location immobilière et de la gestion locative.",
            image:
                "https://images.unsplash.com/photo-1438761681033-6461ffad8d80?ixlib=rb-=eyJhcHBfaWQiOjEyMDd9&auto=format&fit=facearea&facepad=8&w=256&h=256&q=80",
        },
        {
            name: "Pierre Lefevre",
            role: "Conseiller Transaction",
            description: "Expert en vente immobilière et en négociation de biens.",
            image:
                "https://images.unsplash.com/photo-1531427186611-ecfd6d936e79?ixlib=rb-=eyJhcHBfaWQiOjEyMDd9&auto=format&fit=facearea&facepad=8&w=256&h=256&q=80",
        },
        {
            name: "Marie Garcia",
            role: "Assistante Administrative",
            description: "Assure le suivi administratif et la gestion des dossiers clients.",
            image:
                "https://images.unsplash.com/photo-1552058544-f2b08422138a?ixlib=rb-=eyJhcHBfaWQiOjEyMDd9&auto=format&fit=facearea&facepad=8&w=256&h=256&q=80",
        },
        {
            name: "Thomas Dubois",
            role: "Juriste Immobilier",
            description: "Conseille et accompagne nos clients sur les aspects juridiques.",
            image:
                "https://images.unsplash.com/photo-1507003211169-0a1dd8a72fbc?ixlib=rb-=eyJhcHBfaWQiOjEyMDd9&auto=format&fit=facearea&facepad=8&w=256&h=256&q=80",
        },
        {
            name: "Julie Bernard",
            role: "Chargée de Communication",
            description: "Responsable de la communication et du marketing de l'agence.",
            image:
                "https://images.unsplash.com/photo-1494790108377-be9c29b88360?ixlib=rb-=eyJhcHBfaWQiOjEyMDd9&auto=format&fit=facearea&facepad=8&w=256&h=256&q=80",
        },
    ]

    const stats = [
        {
            label: "Transactions réalisées",
            value: "+500",
        },
        {
            label: "Taux de satisfaction",
            value: "98%",
        },
        {
            label: "Délai moyen de location",
            value: "15 jours",
        },
        {
            label: "Taux de recouvrement",
            value: "99%",
        },
    ]

    const testimonials = [
        {
            name: "Sophie L.",
            location: "Paris",
            text: "Grâce à DOGES, j'ai trouvé le bien idéal en un temps record. Leur équipe est très professionnelle et à l'écoute de mes besoins.",
            image:
                "https://images.unsplash.com/photo-1494790108377-be9c29b88360?ixlib=rb-=eyJhcHBfaWQiOjEyMDd9&auto=format&fit=facearea&facepad=8&w=256&h=256&q=80",
        },
        {
            name: "Jean-Pierre M.",
            location: "Marseille",
            text: "Je suis très satisfait des services de gestion locative de DOGES. Ils s'occupent de tout et me permettent de louer mon bien en toute sérénité.",
            image:
                "https://images.unsplash.com/photo-1507003211169-0a1dd8a72fbc?ixlib=rb-=eyJhcHBfaWQiOjEyMDd9&auto=format&fit=facearea&facepad=8&w=256&h=256&q=80",
        },
        {
            name: "Isabelle R.",
            location: "Lyon",
            text: "J'ai confié la vente de mon appartement à DOGES et j'ai été très bien conseillée. Ils ont trouvé un acheteur rapidement et au meilleur prix.",
            image:
                "https://images.unsplash.com/photo-1552058544-f2b08422138a?ixlib=rb-=eyJhcHBfaWQiOjEyMDd9&auto=format&fit=facearea&facepad=8&w=256&h=256&q=80",
        },
    ]

    const faqs = [
        {
            question: "Quels sont vos honoraires de gestion locative ?",
            answer:
                "Nos honoraires de gestion locative varient en fonction du type de bien et des services inclus. Contactez-nous pour obtenir un devis personnalisé.",
        },
        {
            question: "Comment puis-je mettre mon bien en location avec DOGES ?",
            answer:
                "Il vous suffit de nous contacter et de nous fournir les informations relatives à votre bien. Nous nous occupons de tout, de la recherche de locataire à la signature du bail.",
        },
        {
            question: "Proposez-vous des assurances pour les propriétaires ?",
            answer:
                "Oui, nous proposons une gamme complète d'assurances pour protéger votre bien et vous couvrir en cas de sinistre.",
        },
        {
            question: "Comment puis-je vous contacter pour une estimation de mon bien ?",
            answer:
                "Vous pouvez nous contacter par téléphone, par email ou via notre formulaire de contact en ligne. Nous serons ravis de vous aider.",
        },
        {
            question: "Quels documents dois-je fournir pour la mise en gestion de mon bien ?",
            answer:
                "Pour la mise en gestion de votre bien, vous devez nous fournir le titre de propriété, les diagnostics techniques, le dernier avis de taxe foncière, et les coordonnées du syndic si le bien est en copropriété.",
        },
        {
            question: "Combien de temps prend la recherche d'un locataire ?",
            answer:
                "Le délai de recherche d'un locataire dépend de plusieurs facteurs comme l'emplacement, le prix et l'état du bien. En moyenne, nous trouvons un locataire en 2 à 4 semaines pour un bien au prix du marché.",
        },
    ]

    const blogPosts = blogArticles.slice(0, 3).map((article) => ({
        title: article.title,
        excerpt: article.description,
        image: article.image,
        date: article.date,
        slug: article.slug,
    }))

    const quickStats = [
        {
            label: "Biens gérés",
            value: "+1000",
            icon: <Building2 className="h-5 w-5 text-white"/>,
        },
        {
            label: "Années d'expérience",
            value: "20+",
            icon: <Clock className="h-5 w-5 text-white"/>,
        },
        {
            label: "Clients satisfaits",
            value: "+95%",
            icon: <Users className="h-5 w-5 text-white"/>,
        },
    ]

    const handleContactSubmit = (e: React.FormEvent) => {
        e.preventDefault()
        // Rediriger vers la page contact pour un traitement complet
        window.location.href = "/contact"
    }

    return (
        <div className="min-h-screen">
            {/* Hero Section */}
            <section className="relative min-h-screen flex items-center justify-center overflow-hidden mt-0 pt-0">
                <motion.div
                    initial={{scale: 1.1}}
                    animate={{scale: 1}}
                    transition={{duration: 1.5}}
                    className="absolute inset-0 z-0"
                >
                    <Image
                        src="https://images.unsplash.com/photo-1600585152220-90363fe7e115?q=80&w=2070&auto=format&fit=crop"
                        alt="Luxury Parisian Apartment"
                        fill
                        className="object-cover"
                        priority
                        unoptimized
                    />
                    <div className="absolute inset-0 bg-gradient-to-b from-black/60 via-black/40 to-black/60"/>
                </motion.div>

                <motion.div
                    initial={{opacity: 0, y: 20}}
                    animate={{opacity: 1, y: 0}}
                    transition={{duration: 1, delay: 0.5}}
                    className="container mx-auto px-4 text-center relative z-10 text-white"
                >
                    <motion.h1
                        initial={{opacity: 0, y: 20}}
                        animate={{opacity: 1, y: 0}}
                        transition={{duration: 0.8, delay: 0.8}}
                        className="text-5xl md:text-7xl font-bold mb-6 leading-tight"
                    >
                        DOGES - <br/>
                        <span className="bg-gradient-to-r from-white to-gray-300 bg-clip-text text-transparent">
              ADMINISTRATEUR DE BIENS
            </span>
                    </motion.h1>
                    <motion.p
                        initial={{opacity: 0}}
                        animate={{opacity: 1}}
                        transition={{duration: 0.8, delay: 1.2}}
                        className="text-xl md:text-2xl mb-12 text-gray-200 max-w-3xl mx-auto"
                    >
                        La confiance et l'expertise au cœur de notre engagement pour votre patrimoine immobilier.
                    </motion.p>
                    <motion.div
                        initial={{opacity: 0, y: 20}}
                        animate={{opacity: 1, y: 0}}
                        transition={{duration: 0.8, delay: 1.5}}
                        className="flex flex-col sm:flex-row justify-center items-center space-y-4 sm:space-y-0 sm:space-x-6"
                    >
                        <Link href="/services">
                            <Button
                                variant="gradient"
                                className="text-white px-8 py-6 text-lg shadow-xl transition-all duration-300 transform hover:scale-105"
                            >
                                DÉCOUVRIR NOS SERVICES
                                <ArrowRight className="ml-2 h-5 w-5"/>
                            </Button>
                        </Link>
                        <Link href="/contact">
                            <Button
                                variant="outline"
                                className="border-2 border-white bg-transparent text-white hover:bg-white hover:text-[#002395] px-8 py-6 text-lg backdrop-blur-sm transition-all duration-300 transform hover:scale-105"
                            >
                                NOUS CONTACTER
                            </Button>
                        </Link>
                    </motion.div>
                </motion.div>

                <div className="absolute bottom-10 left-1/2 transform -translate-x-1/2 z-10">
                    <motion.div
                        animate={{y: [0, 10, 0]}}
                        transition={{duration: 1.5, repeat: Number.POSITIVE_INFINITY}}
                        className="cursor-pointer"
                    >
                        <ChevronRight size={40} className="text-white rotate-90"/>
                    </motion.div>
                </div>

                {/* Floating Stats */}
                <motion.div
                    initial={{opacity: 0, x: -100}}
                    animate={{opacity: 1, x: 0}}
                    transition={{duration: 0.8, delay: 2}}
                    className="absolute left-4 bottom-20 z-10 hidden lg:block"
                >
                    <div className="bg-black/30 backdrop-blur-md rounded-2xl p-6 border border-white/20">
                        <div className="space-y-4">
                            {quickStats.map((stat, index) => (
                                <div key={index} className="flex items-center space-x-3 text-white">
                                    {stat.icon}
                                    <div>
                                        <div className="text-sm opacity-75">{stat.label}</div>
                                        <div className="font-bold">{stat.value}</div>
                                    </div>
                                </div>
                            ))}
                        </div>
                    </div>
                </motion.div>
            </section>

            {/* Services Section avec optimisations */}
            <section className="py-20 bg-white">
                <div className="container mx-auto px-4">
                    <motion.div
                        initial={{opacity: 0, y: 20}}
                        whileInView={{opacity: 1, y: 0}}
                        viewport={{once: true}}
                        className="text-center mb-16"
                    >
                        <h2 className="text-4xl md:text-5xl font-bold mb-4 bg-gradient-to-r from-[#002395] to-[#001875] bg-clip-text text-transparent">
                            Nos Services
                        </h2>
                        <p className="text-xl text-gray-600 max-w-3xl mx-auto">
                            Des solutions sur mesure pour valoriser et sécuriser votre patrimoine immobilier
                        </p>
                    </motion.div>

                    <div className="grid md:grid-cols-2 lg:grid-cols-4 gap-8">
                        {services.map((service, index) => (
                            <ServiceCardOptimized
                                key={index}
                                title={service.title}
                                description={service.description}
                                image={service.image}
                                link={service.link}
                                index={index}
                                // Ne pas passer isImageLoaded pour l'instant
                                // isImageLoaded={serviceImagesStatus[service.image]?.loaded}
                            />
                        ))}
                    </div>
                </div>
            </section>

            {/* À propos Section */}
            <section className="py-20 bg-gray-50">
                <div className="container mx-auto px-4">
                    <motion.div
                        initial={{opacity: 0, y: 20}}
                        whileInView={{opacity: 1, y: 0}}
                        viewport={{once: true}}
                        className="grid lg:grid-cols-2 gap-12 items-center"
                    >
                        <div className="relative h-[600px] rounded-2xl overflow-hidden">
                            <Image
                                src="https://images.unsplash.com/photo-1497366216548-37526070297c?q=80&w=2069&auto=format&fit=crop"
                                alt="À propos de DOGES"
                                fill
                                className="object-cover"
                                unoptimized
                            />
                        </div>
                        <div className="space-y-6">
                            <h2 className="text-4xl md:text-5xl font-bold bg-gradient-to-r from-[#002395] to-[#001875] bg-clip-text text-transparent">
                                Plus de 20 ans d'expertise à votre service
                            </h2>
                            <p className="text-xl text-gray-600">
                                Depuis notre création, DOGES s'est imposé comme un acteur majeur de l'administration de
                                biens à Paris.
                                Notre expertise, notre professionnalisme et notre engagement envers l'excellence nous
                                ont permis de
                                gagner la confiance de nombreux propriétaires.
                            </p>
                            <div className="space-y-4">
                                <div className="flex items-start space-x-4">
                                    <div className="bg-[#002395] p-2 rounded-lg">
                                        <Building2 className="h-6 w-6 text-white"/>
                                    </div>
                                    <div>
                                        <h3 className="text-xl font-semibold mb-2">Une expertise reconnue</h3>
                                        <p className="text-gray-600">
                                            Plus de 1000 biens gérés avec succès et une connaissance approfondie du
                                            marché immobilier
                                            parisien.
                                        </p>
                                    </div>
                                </div>
                                <div className="flex items-start space-x-4">
                                    <div className="bg-[#002395] p-2 rounded-lg">
                                        <Users className="h-6 w-6 text-white"/>
                                    </div>
                                    <div>
                                        <h3 className="text-xl font-semibold mb-2">Une équipe dédiée</h3>
                                        <p className="text-gray-600">
                                            Des professionnels qualifiés et passionnés, à votre écoute pour répondre à
                                            tous vos besoins.
                                        </p>
                                    </div>
                                </div>
                                <div className="flex items-start space-x-4">
                                    <div className="bg-[#002395] p-2 rounded-lg">
                                        <Shield className="h-6 w-6 text-white"/>
                                    </div>
                                    <div>
                                        <h3 className="text-xl font-semibold mb-2">Des garanties solides</h3>
                                        <p className="text-gray-600">
                                            Une protection optimale de votre patrimoine avec nos solutions d'assurance
                                            sur mesure.
                                        </p>
                                    </div>
                                </div>
                            </div>
                            <Link href="/a-propos">
                                <Button variant="gradient" className="text-white px-8 py-6 text-lg">
                                    En savoir plus sur DOGES
                                    <ArrowRight className="ml-2 h-5 w-5"/>
                                </Button>
                            </Link>
                        </div>
                    </motion.div>
                </div>
            </section>

            {/* Stats Section */}
            <section className="py-20 bg-[#002395] text-white">
                <div className="container mx-auto px-4">
                    <div className="grid md:grid-cols-4 gap-8">
                        {stats.map((stat, index) => (
                            <motion.div
                                key={index}
                                initial={{opacity: 0, y: 20}}
                                whileInView={{opacity: 1, y: 0}}
                                viewport={{once: true}}
                                transition={{delay: index * 0.2}}
                                className="text-center"
                            >
                                <div className="text-4xl font-bold mb-2">{stat.value}</div>
                                <div className="text-blue-200">{stat.label}</div>
                            </motion.div>
                        ))}
                    </div>
                </div>
            </section>

            {/* Témoignages */}
            <section className="py-20 bg-gray-50">
                <div className="container mx-auto px-4">
                    <motion.div
                        initial={{opacity: 0, y: 20}}
                        whileInView={{opacity: 1, y: 0}}
                        viewport={{once: true}}
                        className="text-center mb-16"
                    >
                        <h2 className="text-4xl md:text-5xl font-bold mb-4 bg-gradient-to-r from-[#002395] to-[#001875] bg-clip-text text-transparent">
                            Ce que disent nos clients
                        </h2>
                        <p className="text-xl text-gray-600 max-w-3xl mx-auto">
                            La satisfaction de nos clients est notre meilleure récompense
                        </p>
                    </motion.div>

                    <div className="grid md:grid-cols-3 gap-8">
                        {testimonials.map((testimonial, index) => (
                            <motion.div
                                key={index}
                                initial={{opacity: 0, y: 20}}
                                whileInView={{opacity: 1, y: 0}}
                                viewport={{once: true}}
                                transition={{delay: index * 0.2}}
                            >
                                <div className="bg-white p-8 rounded-2xl shadow-lg">
                                    <Quote className="h-8 w-8 text-[#002395] mb-4"/>
                                    <p className="text-gray-600 mb-6">{testimonial.text}</p>
                                    <div>
                                        <h4 className="font-semibold">{testimonial.name}</h4>
                                        <p className="text-sm text-gray-500">{testimonial.location}</p>
                                    </div>
                                </div>
                            </motion.div>
                        ))}
                    </div>
                </div>
            </section>

            {/* FAQ */}
            <section className="py-20 bg-white">
                <div className="container mx-auto px-4">
                    <motion.div
                        initial={{opacity: 0, y: 20}}
                        whileInView={{opacity: 1, y: 0}}
                        viewport={{once: true}}
                        className="text-center mb-16"
                    >
                        <h2 className="text-4xl md:text-5xl font-bold mb-4 bg-gradient-to-r from-[#002395] to-[#001875] bg-clip-text text-transparent">
                            Questions Fréquentes
                        </h2>
                        <p className="text-xl text-gray-600 max-w-3xl mx-auto">Tout ce que vous devez savoir sur nos
                            services</p>
                    </motion.div>

                    <div className="grid md:grid-cols-2 gap-8 max-w-5xl mx-auto">
                        {faqs.map((faq, index) => (
                            <motion.div
                                key={index}
                                initial={{opacity: 0, y: 20}}
                                whileInView={{opacity: 1, y: 0}}
                                viewport={{once: true}}
                                transition={{delay: index * 0.1}}
                                className="mb-6"
                            >
                                <div className="bg-gray-50 p-6 rounded-2xl h-full">
                                    <h3 className="text-lg font-semibold mb-3">{faq.question}</h3>
                                    <p className="text-gray-600">{faq.answer}</p>
                                </div>
                            </motion.div>
                        ))}
                    </div>
                </div>
            </section>

            {/* Blog/Actualités */}
            <section className="py-20 bg-gray-50">
                <div className="container mx-auto px-4">
                    <motion.div
                        initial={{opacity: 0, y: 20}}
                        whileInView={{opacity: 1, y: 0}}
                        viewport={{once: true}}
                        className="text-center mb-16"
                    >
                        <h2 className="text-4xl md:text-5xl font-bold mb-4 bg-gradient-to-r from-[#002395] to-[#001875] bg-clip-text text-transparent">
                            Actualités Immobilières
                        </h2>
                        <p className="text-xl text-gray-600 max-w-3xl mx-auto">Restez informé des dernières tendances du
                            marché</p>
                    </motion.div>

                    <div className="grid md:grid-cols-3 gap-8">
                        {blogPosts.map((post, index) => (
                            <motion.div
                                key={index}
                                initial={{opacity: 0, y: 20}}
                                whileInView={{opacity: 1, y: 0}}
                                viewport={{once: true}}
                                transition={{delay: index * 0.2}}
                                className="group cursor-pointer"
                            >
                                <Link href={`/blog/articles/${post.slug}`} className="block">
                                    <div
                                        className="bg-white rounded-2xl overflow-hidden shadow-lg hover:shadow-2xl transition-all duration-500">
                                        <div className="relative h-48">
                                            <Image
                                                src={post.image || "/placeholder.svg"}
                                                alt={post.title}
                                                fill
                                                className="object-cover"
                                                unoptimized
                                            />
                                        </div>
                                        <div className="p-6">
                                            <div className="flex items-center text-sm text-gray-500 mb-3">
                                                <Calendar className="h-4 w-4 mr-2"/>
                                                {post.date}
                                            </div>
                                            <h3 className="text-xl font-semibold mb-3">{post.title}</h3>
                                            <p className="text-gray-600 mb-4 line-clamp-2">{post.excerpt}</p>
                                            <Button variant="link" className="text-[#002395] p-0 hover:text-[#001875]">
                                                Lire la suite
                                                <ArrowRight className="ml-2 h-4 w-4"/>
                                            </Button>
                                        </div>
                                    </div>
                                </Link>
                            </motion.div>
                        ))}
                    </div>
                </div>
            </section>

            {/* Section Zones d'intervention pour SEO */}
            <section className="py-16 bg-gray-50">
                <div className="container mx-auto px-4">
                    <motion.div
                        initial={{opacity: 0, y: 20}}
                        whileInView={{opacity: 1, y: 0}}
                        viewport={{once: true}}
                        className="text-center mb-12"
                    >
                        <h2 className="text-4xl md:text-5xl font-bold mb-4 bg-gradient-to-r from-[#002395] to-[#001875] bg-clip-text text-transparent">
                            Nos zones d'intervention
                        </h2>
                        <p className="text-xl text-gray-600 max-w-3xl mx-auto">
                            DOGES vous accompagne dans la gestion de votre patrimoine immobilier partout en
                            Île-de-France
                        </p>
                    </motion.div>

                    <div className="mb-12">
                        <h3 className="text-2xl font-semibold mb-6 text-center">Gestion locative à Paris</h3>
                        <div className="flex flex-wrap justify-center gap-3 mb-8">
                            {[
                                "Gestion locative Paris 1er",
                                "Gestion locative Paris 2ème",
                                "Gestion locative Paris 3ème",
                                "Gestion locative Paris 4ème",
                                "Gestion locative Paris 5ème",
                                "Gestion locative Paris 6ème",
                                "Gestion locative Paris 7ème",
                                "Gestion locative Paris 8ème",
                                "Gestion locative Paris 9ème",
                                "Gestion locative Paris 10ème",
                                "Gestion locative Paris 11ème",
                                "Gestion locative Paris 12ème",
                                "Gestion locative Paris 13ème",
                                "Gestion locative Paris 14ème",
                                "Gestion locative Paris 15ème",
                                "Gestion locative Paris 16ème",
                                "Gestion locative Paris 17ème",
                                "Gestion locative Paris 18ème",
                                "Gestion locative Paris 19ème",
                                "Gestion locative Paris 20ème",
                            ].map((zone, index) => {
                                // Extraire le numéro d'arrondissement
                                const arrondissement = zone
                                    .replace("Gestion locative Paris ", "")
                                    .replace(/[èêé]/g, "e")
                                    .replace(/[èêé]/g, "e")
                                // Formater l'arrondissement pour l'URL
                                const arrondissementUrl = arrondissement.toLowerCase()

                                return (
                                    <Link
                                        href={`/services/gestion-locative-paris/${arrondissementUrl}`}
                                        key={index}
                                        className="bg-white text-gray-800 text-sm px-4 py-2 rounded border border-gray-200 hover:bg-gray-50"
                                    >
                                        {zone}
                                    </Link>
                                )
                            })}
                        </div>
                    </div>

                    <div className="mb-12">
                        <h3 className="text-2xl font-semibold mb-6 text-center">Gestion locative en Île-de-France</h3>
                        <div className="flex flex-wrap justify-center gap-3 mb-8">
                            {[
                                "Gestion locative Neuilly-sur-Seine",
                                "Gestion locative Levallois-Perret",
                                "Gestion locative Boulogne-Billancourt",
                                "Gestion locative Issy-les-Moulineaux",
                                "Gestion locative Saint-Cloud",
                                "Gestion locative Sèvres",
                                "Gestion locative Versailles",
                                "Gestion locative Saint-Germain-en-Laye",
                                "Gestion locative Rueil-Malmaison",
                                "Gestion locative Courbevoie",
                                "Gestion locative Puteaux",
                                "Gestion locative Suresnes",
                                "Gestion locative Nanterre",
                                "Gestion locative Vincennes",
                                "Gestion locative Saint-Mandé",
                                "Gestion locative Montreuil",
                                "Gestion locative Asnières-sur-Seine",
                                "Gestion locative Clichy",
                                "Gestion locative La Défense",
                                "Gestion locative Antony",
                            ].map((zone, index) => {
                                // Extraire le nom de la ville (après "Gestion locative ")
                                const ville = zone.replace("Gestion locative ", "")
                                // Formater le nom de la ville pour l'URL (minuscules, tirets)
                                const villeUrl = ville
                                    .toLowerCase()
                                    .replace(/ /g, "-")
                                    .normalize("NFD")
                                    .replace(/[\u0300-\u036f]/g, "")

                                return (
                                    <Link
                                        href={`/services/gestion-locative-ile-de-france/${villeUrl}`}
                                        key={index}
                                        className="bg-white text-gray-800 text-sm px-4 py-2 rounded border border-gray-200 hover:bg-gray-50"
                                    >
                                        {zone}
                                    </Link>
                                )
                            })}
                        </div>
                    </div>

                    <div className="mb-12">
                        <h3 className="text-2xl font-semibold mb-6 text-center">Administrateur de biens en
                            Île-de-France</h3>
                        <div className="flex flex-wrap justify-center gap-3">
                            {[
                                "Administrateur de biens Paris",
                                "Administrateur de biens Hauts-de-Seine",
                                "Administrateur de biens Yvelines",
                                "Administrateur de biens Val-de-Marne",
                                "Administrateur de biens Seine-Saint-Denis",
                                "Administrateur de biens Val-d'Oise",
                                "Administrateur de biens Essonne",
                                "Administrateur de biens Seine-et-Marne",
                                "Administrateur de biens Créteil",
                                "Administrateur de biens Boulogne-Billancourt",
                                "Administrateur de biens Versailles",
                                "Administrateur de biens Nanterre",
                                "Administrateur de biens Évry",
                                "Administrateur de biens Cergy",
                                "Administrateur de biens Bobigny",
                                "Administrateur de biens Melun",
                            ].map((zone, index) => (
                                <Link
                                    href={`/services/administrateur-biens-paris`}
                                    key={index}
                                    className="bg-white text-gray-800 text-sm px-4 py-2 rounded border border-gray-200 hover:bg-gray-50"
                                >
                                    {zone}
                                </Link>
                            ))}
                        </div>
                    </div>
                </div>
            </section>

            {/* National Presence Section */}
            <NationalPresence/>

            {/* Contact Section */}
            <section className="py-24 bg-[#001875] text-white">
                <div className="container mx-auto px-4">
                    <div className="grid md:grid-cols-2 gap-12 items-center">
                        <div className="space-y-6">
                            <h2 className="text-4xl md:text-5xl font-bold">Contactez-nous</h2>
                            <p className="text-xl">
                                Notre équipe est à votre disposition pour répondre à toutes vos questions et vous
                                accompagner dans vos
                                projets immobiliers.
                            </p>
                            <div className="space-y-4">
                                <div className="flex items-center space-x-4">
                                    <MapPin className="h-6 w-6 text-blue-200"/>
                                    <p>92, Avenue de Saint-Mandé Paris 12e</p>
                                </div>
                                <div className="flex items-center space-x-4">
                                    <Clock className="h-6 w-6 text-blue-200"/>
                                    <p>Du lundi au vendredi, de 9h à 18h</p>
                                </div>
                                <div className="flex items-center space-x-4">
                                    <MessageCircle className="h-6 w-6 text-blue-200"/>
                                    <p>contact@dogesadb.fr</p>
                                </div>
                            </div>
                            <div className="flex space-x-4">
                                <Link href="#" className="hover:text-blue-200 transition-colors">
                                    <Facebook className="h-6 w-6"/>
                                </Link>
                                <Link href="#" className="hover:text-blue-200 transition-colors">
                                    <Twitter className="h-6 w-6"/>
                                </Link>
                                <Link href="#" className="hover:text-blue-200 transition-colors">
                                    <Linkedin className="h-6 w-6"/>
                                </Link>
                                <Link href="#" className="hover:text-blue-200 transition-colors">
                                    <Instagram className="h-6 w-6"/>
                                </Link>
                            </div>
                        </div>
                        <div>
                            <form className="space-y-6" onSubmit={handleContactSubmit}>
                                <div>
                                    <label htmlFor="name" className="block text-sm font-medium text-blue-200">
                                        Nom
                                    </label>
                                    <input
                                        type="text"
                                        id="name"
                                        name="name"
                                        className="mt-1 block w-full py-3 px-4 bg-white/10 border-none rounded-md text-white placeholder:text-blue-200 focus:ring-2 focus:ring-blue-500 focus:outline-none"
                                        placeholder="Votre nom"
                                        required
                                    />
                                </div>
                                <div>
                                    <label htmlFor="email" className="block text-sm font-medium text-blue-200">
                                        Email
                                    </label>
                                    <input
                                        type="email"
                                        id="email"
                                        name="email"
                                        className="mt-1 block w-full py-3 px-4 bg-white/10 border-none rounded-md text-white placeholder:text-blue-200 focus:ring-2 focus:ring-blue-500 focus:outline-none"
                                        placeholder="Votre email"
                                        required
                                    />
                                </div>
                                <div>
                                    <label htmlFor="message" className="block text-sm font-medium text-blue-200">
                                        Message
                                    </label>
                                    <textarea
                                        id="message"
                                        name="message"
                                        rows={4}
                                        className="mt-1 block w-full py-3 px-4 bg-white/10 border-none rounded-md text-white placeholder:text-blue-200 focus:ring-2 focus:ring-blue-500 focus:outline-none"
                                        placeholder="Votre message"
                                        required
                                    />
                                </div>
                                <Button type="submit" variant="gradient"
                                        className="w-full text-white px-8 py-4 text-lg">
                                    Envoyer le message
                                    <ArrowRight className="ml-2 h-5 w-5"/>
                                </Button>
                            </form>
                        </div>
                    </div>
                </div>
            </section>
        </div>
    )
}
