import { Check, Shield } from "lucide-react"
import Image from "next/image"
import type { Metadata } from "next"
import SmartContactForm from "@/components/smart-contact-form"
import ServiceHeader from "@/components/service-header"
import TableOfContentsWidget from "@/components/table-of-contents-widget"
import ServicesSidebarWidgets from "@/components/services-sidebar-widgets"
import ContactWidget from "@/components/contact-widget"

// Métadonnées SEO optimisées
export const metadata: Metadata = {
  title: "Assurance Loyers Impayés | DOGES Immobilier",
  description:
    "Protégez vos revenus locatifs avec notre assurance loyers impayés. Couverture complète contre les impayés, dégradations et frais de procédure.",
  keywords: "assurance loyers impayés, GLI, garantie loyers impayés, protection propriétaire, impayés locatifs",
}

export default function AssuranceLoyersImpayes() {
  // Définition des éléments de la table des matières
  const tableOfContentsItems = [
    { id: "presentation", text: "Qu'est-ce que l'assurance loyers impayés ?", level: 2 },
    { id: "avantages", text: "Les avantages de notre assurance loyers impayés", level: 2 },
    { id: "couverture", text: "Ce que couvre notre assurance loyers impayés", level: 2 },
    { id: "fonctionnement", text: "Comment fonctionne notre assurance loyers impayés ?", level: 2 },
    { id: "faq", text: "Questions fréquentes sur l'assurance loyers impayés", level: 2 },
  ]

  return (
    <>
      <ServiceHeader
        title="Assurance Loyers Impayés"
        description="Protégez vos revenus locatifs et sécurisez votre investissement immobilier"
        breadcrumbItems={[
          { name: "Accueil", url: "/" },
          { name: "Services", url: "/services" },
          { name: "Assurance Loyers Impayés", url: "/services/assurance-loyers-impayes" },
        ]}
        backgroundImage="/rental-income-protection-security.png"
      />

      {/* Structured data */}
      <script
        type="application/ld+json"
        dangerouslySetInnerHTML={{
          __html: JSON.stringify({
            "@context": "https://schema.org",
            "@type": "Service",
            serviceType: "Assurance Loyers Impayés",
            provider: {
              "@type": "RealEstateAgent",
              name: "DOGES",
              url: "https://www.doges.fr",
            },
            description:
              "Protégez vos revenus locatifs avec notre assurance loyers impayés. Couverture complète contre les impayés, dégradations et frais de procédure.",
            offers: {
              "@type": "Offer",
              priceSpecification: {
                "@type": "PriceSpecification",
                price: "2.5-3.5",
                priceCurrency: "EUR",
                valueAddedTaxIncluded: false,
                unitText: "% du montant annuel des loyers et charges",
              },
            },
            termsOfService:
              "Conditions d'éligibilité : taux d'effort du locataire inférieur à 33%, CDI hors période d'essai ou statut stable",
          }),
        }}
      />

      <div className="container mx-auto px-4 py-12">
        <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
          {/* Contenu principal */}
          <div className="lg:col-span-2">
            <section className="mb-12" id="presentation">
              <div className="grid lg:grid-cols-2 gap-12 items-center">
                <div>
                  <h2 className="text-3xl font-bold mb-6 text-[#002395]">Qu'est-ce que l'assurance loyers impayés ?</h2>
                  <p className="text-lg text-gray-700 mb-6">
                    L'assurance loyers impayés (GLI) est une protection essentielle pour tout propriétaire bailleur.
                    Elle vous garantit le versement des loyers en cas de défaillance du locataire et vous protège contre
                    les risques locatifs.
                  </p>
                  <p className="text-lg text-gray-700 mb-6">
                    Chez DOGES, nous proposons une solution d'assurance complète pour sécuriser vos revenus locatifs et
                    vous permettre de profiter sereinement de votre investissement immobilier.
                  </p>
                  <div className="space-y-4">
                    {[
                      "Prise en charge des loyers impayés",
                      "Couverture des frais de procédure",
                      "Indemnisation des dégradations locatives",
                      "Protection juridique",
                      "Garantie du maintien de vos revenus locatifs",
                    ].map((feature, index) => (
                      <div key={index} className="flex items-start">
                        <div className="mt-1 mr-3 bg-[#002395] rounded-full p-1">
                          <Check className="h-4 w-4 text-white" />
                        </div>
                        <p>{feature}</p>
                      </div>
                    ))}
                  </div>
                </div>
                <div className="relative h-[400px] rounded-2xl overflow-hidden">
                  <Image
                    src="https://images.unsplash.com/photo-1560520653-9e0e4c89eb11?q=80&w=2073&auto=format&fit=crop"
                    alt="Assurance loyers impayés"
                    fill
                    className="object-cover"
                  />
                </div>
              </div>
            </section>

            {/* Avantages de l'assurance loyers impayés */}
            <section className="py-12 bg-gray-50 -mx-4 px-4 lg:-mx-8 lg:px-8 mb-12" id="avantages">
              <h2 className="text-3xl font-bold mb-12 text-center text-[#002395]">
                Les avantages de notre assurance loyers impayés
              </h2>
              <div className="grid md:grid-cols-3 gap-8">
                {[
                  {
                    title: "Sécurité financière",
                    description:
                      "Garantie du paiement des loyers même en cas de défaillance du locataire, préservant ainsi vos revenus locatifs.",
                    icon: <Shield className="h-10 w-10 text-white" />,
                  },
                  {
                    title: "Tranquillité d'esprit",
                    description:
                      "Gestion complète des procédures de recouvrement et d'expulsion en cas d'impayés, sans stress pour vous.",
                    icon: <Shield className="h-10 w-10 text-white" />,
                  },
                  {
                    title: "Protection complète",
                    description:
                      "Couverture des dégradations immobilières causées par le locataire et des frais de contentieux.",
                    icon: <Shield className="h-10 w-10 text-white" />,
                  },
                ].map((advantage, index) => (
                  <div key={index} className="bg-white rounded-xl shadow-lg p-8 border border-gray-100">
                    <div className="bg-gradient-to-r from-[#002395] to-[#001875] w-16 h-16 rounded-full flex items-center justify-center mb-6">
                      {advantage.icon}
                    </div>
                    <h3 className="text-xl font-bold mb-4">{advantage.title}</h3>
                    <p className="text-gray-700">{advantage.description}</p>
                  </div>
                ))}
              </div>
            </section>

            {/* Ce que couvre notre assurance */}
            <section className="mb-12" id="couverture">
              <h2 className="text-3xl font-bold mb-12 text-center text-[#002395]">
                Ce que couvre notre assurance loyers impayés
              </h2>
              <div className="grid md:grid-cols-2 gap-8">
                {[
                  {
                    title: "Loyers impayés",
                    description:
                      "Prise en charge des loyers et charges impayés jusqu'à 24 mois, avec un plafond mensuel adapté à votre bien.",
                    included: true,
                  },
                  {
                    title: "Frais de procédure",
                    description:
                      "Couverture des frais d'huissier, d'avocat et de procédure d'expulsion en cas de contentieux avec le locataire.",
                    included: true,
                  },
                  {
                    title: "Dégradations locatives",
                    description:
                      "Indemnisation des dommages causés par le locataire au-delà du dépôt de garantie, jusqu'à 10 000€.",
                    included: true,
                  },
                  {
                    title: "Protection juridique",
                    description:
                      "Assistance juridique pour tous les litiges liés à la location de votre bien immobilier.",
                    included: true,
                  },
                  {
                    title: "Vacance locative",
                    description:
                      "Indemnisation en cas de vacance locative suite à un sinistre couvert par l'assurance.",
                    included: false,
                  },
                  {
                    title: "Défense-recours",
                    description:
                      "Prise en charge des frais de défense de vos intérêts en cas de litige avec le locataire.",
                    included: true,
                  },
                ].map((coverage, index) => (
                  <div key={index} className="bg-white rounded-xl shadow-sm p-6 border border-gray-100">
                    <div className="flex items-center mb-4">
                      <div
                        className={`w-8 h-8 rounded-full flex items-center justify-center mr-3 ${coverage.included ? "bg-green-100 text-green-600" : "bg-gray-100 text-gray-400"}`}
                      >
                        {coverage.included ? <Check className="h-5 w-5" /> : "X"}
                      </div>
                      <h3 className="text-xl font-semibold">{coverage.title}</h3>
                    </div>
                    <p className="text-gray-700">{coverage.description}</p>
                  </div>
                ))}
              </div>
            </section>

            {/* Comment ça marche */}
            <section className="py-12 bg-gray-50 -mx-4 px-4 lg:-mx-8 lg:px-8 mb-12" id="fonctionnement">
              <h2 className="text-3xl font-bold mb-12 text-center text-[#002395]">
                Comment fonctionne notre assurance loyers impayés ?
              </h2>
              <div className="max-w-3xl mx-auto">
                <div className="space-y-8">
                  {[
                    {
                      step: "1",
                      title: "Sélection du locataire",
                      description:
                        "Nous vérifions la solvabilité du locataire selon les critères de l'assurance pour garantir l'éligibilité à la GLI.",
                    },
                    {
                      step: "2",
                      title: "Souscription de l'assurance",
                      description:
                        "Une fois le locataire validé, nous souscrivons l'assurance loyers impayés pour votre bien.",
                    },
                    {
                      step: "3",
                      title: "Suivi des paiements",
                      description:
                        "Nous assurons un suivi rigoureux des paiements des loyers et intervenons dès le premier retard.",
                    },
                    {
                      step: "4",
                      title: "Gestion des impayés",
                      description:
                        "En cas d'impayé, nous déclenchons les procédures de recouvrement et activons l'assurance pour maintenir vos revenus.",
                    },
                    {
                      step: "5",
                      title: "Indemnisation",
                      description:
                        "Vous recevez les indemnisations pour les loyers impayés pendant toute la durée de la procédure, jusqu'à 24 mois.",
                    },
                  ].map((item) => (
                    <div key={item.step} className="flex">
                      <div className="mr-6">
                        <div className="w-10 h-10 rounded-full bg-[#002395] text-white flex items-center justify-center font-bold">
                          {item.step}
                        </div>
                      </div>
                      <div>
                        <h4 className="text-xl font-medium mb-2">{item.title}</h4>
                        <p className="text-gray-700">{item.description}</p>
                      </div>
                    </div>
                  ))}
                </div>
              </div>
            </section>

            {/* FAQ pour le SEO */}
            <section className="mb-12" id="faq">
              <h2 className="text-3xl font-bold mb-12 text-center text-[#002395]">
                Questions fréquentes sur l'assurance loyers impayés
              </h2>
              <div className="space-y-6">
                {[
                  {
                    question: "Quel est le coût d'une assurance loyers impayés ?",
                    answer:
                      "Le coût d'une assurance loyers impayés varie généralement entre 2,5% et 3,5% du montant annuel des loyers et charges. Ce tarif dépend de plusieurs facteurs comme le type de bien, sa localisation et le profil du locataire. Chez DOGES, nous négocions les meilleures conditions avec nos partenaires assureurs pour vous offrir un rapport qualité-prix optimal.",
                  },
                  {
                    question: "Quelles sont les conditions pour bénéficier de l'assurance loyers impayés ?",
                    answer:
                      "Pour bénéficier de l'assurance loyers impayés, le locataire doit généralement présenter un taux d'effort (ratio loyer/revenus) inférieur à 33%, être en CDI hors période d'essai ou avoir un statut stable (fonctionnaire, retraité), et ne pas avoir d'incidents de paiement récents. Notre équipe vérifie rigoureusement ces critères pour garantir l'éligibilité à l'assurance.",
                  },
                  {
                    question: "L'assurance loyers impayés est-elle obligatoire ?",
                    answer:
                      "Non, l'assurance loyers impayés n'est pas obligatoire, mais elle est fortement recommandée pour tout propriétaire bailleur souhaitant sécuriser ses revenus locatifs. Elle offre une protection complète contre les risques d'impayés, les dégradations et les frais de procédure, vous permettant de profiter sereinement de votre investissement immobilier.",
                  },
                ].map((item, index) => (
                  <div key={index} className="bg-white rounded-lg shadow-sm p-6 border border-gray-100">
                    <h3 className="text-lg font-semibold mb-2">{item.question}</h3>
                    <p className="text-gray-700">{item.answer}</p>
                  </div>
                ))}
              </div>
            </section>

            {/* Formulaire de contact */}
            <section className="py-12 bg-gray-50 -mx-4 px-4 lg:-mx-8 lg:px-8">
              <div className="max-w-3xl mx-auto">
                <h2 className="text-3xl font-bold mb-8 text-center text-[#002395]">
                  Demandez un devis pour votre assurance loyers impayés
                </h2>
                <SmartContactForm />
              </div>
            </section>
          </div>

          {/* Sidebar */}
          <div className="lg:col-span-1">
            <div className="sticky top-24 space-y-8">
              {/* Widget Table des matières */}
              <TableOfContentsWidget title="Cette page" items={tableOfContentsItems} />

              {/* Widget Services associés */}
              <ServicesSidebarWidgets />

              {/* Widget Contact rapide */}
              <ContactWidget/>
            </div>
          </div>
        </div>
      </div>
    </>
  )
}
