import { Shield, Check, FileText, Banknote, Clock, Home, Umbrella } from "lucide-react"
import Image from "next/image"
import type { Metadata } from "next"

import ServiceHeader from "@/components/service-header"
import SmartContactForm from "@/components/smart-contact-form"
import TableOfContentsWidget from "@/components/table-of-contents-widget"
import ContactWidget from "@/components/contact-widget"
import RelatedServicesWidget from "@/components/related-services-widget"

// Métadonnées SEO optimisées
export const metadata: Metadata = {
  title: "Assurance PNO (Propriétaire Non Occupant) | DOGES Immobilier",
  description:
    "Protégez votre bien immobilier avec notre assurance propriétaire non occupant (PNO). Couverture complète contre les risques locatifs et protection de votre patrimoine.",
  keywords: "assurance PNO, propriétaire non occupant, assurance habitation, protection propriétaire, risques locatifs",
}

export default function AssurancePNO() {
  // Définir les éléments de la table des matières
  const tocItems = [
    { id: "presentation", title: "Qu'est-ce que l'assurance PNO ?" },
    { id: "couverture", title: "Ce que couvre l'assurance PNO" },
    { id: "avantages", title: "Pourquoi souscrire une assurance PNO" },
    { id: "formules", title: "Nos formules d'assurance" },
    { id: "faq", title: "Questions fréquentes" },
    { id: "temoignages", title: "Témoignages clients" },
  ]

  // Définir les services associés
  const relatedServices = [
    {
      title: "Gestion Locative",
      description: "Confiez-nous la gestion complète de votre bien immobilier",
      link: "/services/gestion-locative",
    },
    {
      title: "Garantie Loyers Impayés",
      description: "Protégez vos revenus locatifs contre les impayés",
      link: "/services/garantie-loyers-impayes",
    },
    {
      title: "Service Juridique",
      description: "Conseil et accompagnement juridique spécialisé",
      link: "/services/juridique",
    },
  ]
  return (
    <>
      <ServiceHeader
        title="Assurance Propriétaire Non Occupant"
        description="Protégez votre bien immobilier et votre responsabilité civile en tant que propriétaire bailleur"
        breadcrumbItems={[
          { name: "Accueil", url: "/" },
          { name: "Services", url: "/services" },
          { name: "Assurance Propriétaire Non Occupant", url: "/services/assurance-pno" },
        ]}
        backgroundImage="/images/assurance-pno-service.webp"
      />

      <div className="container mx-auto px-4 py-12">
        <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
          {/* Contenu principal */}
          <div className="lg:col-span-2">
            {/* Section d'introduction */}
            <section className="mb-12" id="presentation">
              <h2 className="text-3xl font-bold mb-6 text-[#002395]">
                Qu'est-ce que l'Assurance Propriétaire Non Occupant ?
              </h2>
              <div className="grid md:grid-cols-2 gap-8 items-center mb-8">
                <div>
                  <p className="text-lg text-gray-700 mb-6">
                    L'Assurance Propriétaire Non Occupant (PNO) est une protection essentielle pour tout propriétaire
                    qui loue son bien immobilier. Elle couvre les risques liés à votre bien lorsque celui-ci est loué ou
                    inoccupé entre deux locations.
                  </p>
                  <p className="text-lg text-gray-700">
                    Chez DOGES, nous proposons une solution d'assurance complète pour protéger votre patrimoine
                    immobilier contre les sinistres et vous prémunir contre les risques de responsabilité civile.
                  </p>
                </div>
                <div className="relative h-64 rounded-xl overflow-hidden shadow-lg">
                  <Image
                    src="/images/assurance-pno-service.webp"
                    alt="Assurance propriétaire non occupant"
                    fill
                    className="object-cover"
                  />
                </div>
              </div>

              <div className="bg-gray-50 p-6 rounded-xl border border-gray-100 mb-8">
                <h3 className="text-xl font-semibold mb-4">Les points clés de notre assurance PNO</h3>
                <div className="grid md:grid-cols-2 gap-4">
                  {[
                    "Protection contre les dégâts des eaux et incendies",
                    "Couverture de la responsabilité civile propriétaire",
                    "Indemnisation en cas de catastrophes naturelles",
                    "Protection juridique incluse",
                    "Garantie des risques locatifs",
                    "Assistance 24h/24 et 7j/7",
                  ].map((feature, index) => (
                    <div key={index} className="flex items-start">
                      <div className="mt-1 mr-3 bg-[#002395] rounded-full p-1">
                        <Check className="h-4 w-4 text-white" />
                      </div>
                      <p>{feature}</p>
                    </div>
                  ))}
                </div>
              </div>
            </section>

            {/* Section des avantages */}
            <section className="mb-12" id="avantages">
              <h2 className="text-3xl font-bold mb-6 text-[#002395]">
                Les avantages de notre Assurance Propriétaire Non Occupant
              </h2>
              <div className="grid md:grid-cols-3 gap-6">
                {[
                  {
                    title: "Protection complète",
                    description:
                      "Couverture étendue contre les principaux risques immobiliers : incendie, dégâts des eaux, catastrophes naturelles et responsabilité civile.",
                    icon: <Shield className="h-10 w-10 text-white" />,
                  },
                  {
                    title: "Tranquillité d'esprit",
                    description:
                      "Sécurisation de votre investissement immobilier avec une protection adaptée aux spécificités des biens mis en location.",
                    icon: <Umbrella className="h-10 w-10 text-white" />,
                  },
                  {
                    title: "Gestion simplifiée",
                    description:
                      "Prise en charge rapide des sinistres et accompagnement personnalisé par nos experts pour faciliter vos démarches.",
                    icon: <Home className="h-10 w-10 text-white" />,
                  },
                ].map((advantage, index) => (
                  <div key={index} className="bg-white rounded-xl shadow-sm p-6 border border-gray-100">
                    <div className="bg-gradient-to-r from-[#002395] to-[#001875] w-16 h-16 rounded-full flex items-center justify-center mb-4">
                      {advantage.icon}
                    </div>
                    <h3 className="text-xl font-bold mb-2">{advantage.title}</h3>
                    <p className="text-gray-700">{advantage.description}</p>
                  </div>
                ))}
              </div>
            </section>

            {/* Section de couverture */}
            <section className="mb-12" id="couverture">
              <h2 className="text-3xl font-bold mb-6 text-[#002395]">
                Ce que couvre notre Assurance Propriétaire Non Occupant
              </h2>
              <div className="grid md:grid-cols-2 gap-6">
                {[
                  {
                    title: "Incendie et risques annexes",
                    description:
                      "Protection contre les incendies, explosions, foudre et dommages électriques affectant votre bien immobilier.",
                    included: true,
                  },
                  {
                    title: "Dégâts des eaux",
                    description:
                      "Couverture des dommages causés par les fuites, infiltrations et débordements d'eau dans votre propriété.",
                    included: true,
                  },
                  {
                    title: "Catastrophes naturelles",
                    description:
                      "Indemnisation en cas de dommages causés par des événements naturels reconnus comme catastrophes par arrêté ministériel.",
                    included: true,
                  },
                  {
                    title: "Responsabilité civile propriétaire",
                    description:
                      "Protection contre les recours des voisins, des tiers et des locataires en cas de dommages causés par votre bien.",
                    included: true,
                  },
                  {
                    title: "Défense et recours",
                    description:
                      "Prise en charge des frais de défense de vos intérêts en cas de litige lié à votre bien immobilier.",
                    included: true,
                  },
                  {
                    title: "Bris de glace",
                    description:
                      "Couverture des dommages aux vitres, fenêtres, baies vitrées et autres éléments vitrés de votre propriété.",
                    included: true,
                  },
                  {
                    title: "Vol et vandalisme",
                    description:
                      "Protection contre les actes de vol, tentative de vol et vandalisme commis dans votre bien immobilier.",
                    included: true,
                  },
                  {
                    title: "Assistance 24h/24",
                    description:
                      "Service d'assistance disponible à tout moment pour vous aider en cas de sinistre ou d'urgence.",
                    included: true,
                  },
                ].map((coverage, index) => (
                  <div key={index} className="bg-white rounded-xl shadow-sm p-6 border border-gray-100">
                    <div className="flex items-center mb-3">
                      <div
                        className={`w-8 h-8 rounded-full flex items-center justify-center mr-3 ${
                          coverage.included ? "bg-green-100 text-green-600" : "bg-gray-100 text-gray-400"
                        }`}
                      >
                        {coverage.included ? <Check className="h-5 w-5" /> : "X"}
                      </div>
                      <h3 className="text-xl font-semibold">{coverage.title}</h3>
                    </div>
                    <p className="text-gray-700">{coverage.description}</p>
                  </div>
                ))}
              </div>
            </section>

            {/* Section de fonctionnement */}
            <section className="mb-12" id="fonctionnement">
              <h2 className="text-3xl font-bold mb-6 text-[#002395]">
                Comment fonctionne notre Assurance Propriétaire Non Occupant ?
              </h2>
              <div className="bg-white rounded-xl shadow-sm p-6 border border-gray-100 mb-8">
                <div className="space-y-8">
                  {[
                    {
                      step: "1",
                      title: "Évaluation de vos besoins",
                      description:
                        "Nous analysons les caractéristiques de votre bien immobilier pour déterminer les garanties adaptées à votre situation.",
                    },
                    {
                      step: "2",
                      title: "Proposition personnalisée",
                      description:
                        "Nous vous proposons une offre d'assurance sur mesure avec différentes options de couverture selon vos besoins.",
                    },
                    {
                      step: "3",
                      title: "Souscription simplifiée",
                      description:
                        "Processus de souscription rapide et sans tracas, avec possibilité de signature électronique des documents.",
                    },
                    {
                      step: "4",
                      title: "Gestion des sinistres",
                      description:
                        "En cas de sinistre, notre équipe vous accompagne dans toutes les démarches et coordonne les interventions nécessaires.",
                    },
                    {
                      step: "5",
                      title: "Suivi et renouvellement",
                      description:
                        "Nous assurons un suivi régulier de votre contrat et vous proposons des ajustements si nécessaire lors du renouvellement.",
                    },
                  ].map((item) => (
                    <div key={item.step} className="flex">
                      <div className="mr-6">
                        <div className="w-10 h-10 rounded-full bg-[#002395] text-white flex items-center justify-center font-bold">
                          {item.step}
                        </div>
                      </div>
                      <div>
                        <h4 className="text-xl font-medium mb-2">{item.title}</h4>
                        <p className="text-gray-700">{item.description}</p>
                      </div>
                    </div>
                  ))}
                </div>
              </div>
            </section>

            {/* Section Pourquoi nous choisir */}
            <section className="mb-12" id="pourquoi-nous">
              <h2 className="text-3xl font-bold mb-6 text-[#002395]">
                Pourquoi choisir DOGES pour votre Assurance Propriétaire Non Occupant ?
              </h2>
              <div className="grid md:grid-cols-2 gap-6">
                {[
                  {
                    title: "Expertise immobilière",
                    description:
                      "Notre connaissance approfondie du marché immobilier nous permet de vous proposer des solutions d'assurance parfaitement adaptées à votre bien.",
                    icon: <Home className="h-6 w-6 text-[#002395]" />,
                  },
                  {
                    title: "Tarifs compétitifs",
                    description:
                      "Grâce à nos partenariats avec les principales compagnies d'assurance, nous vous garantissons les meilleurs tarifs pour une couverture optimale.",
                    icon: <Banknote className="h-6 w-6 text-[#002395]" />,
                  },
                  {
                    title: "Service personnalisé",
                    description:
                      "Un conseiller dédié vous accompagne tout au long de votre contrat pour répondre à vos questions et vous aider en cas de besoin.",
                    icon: <FileText className="h-6 w-6 text-[#002395]" />,
                  },
                  {
                    title: "Gestion réactive des sinistres",
                    description:
                      "Notre équipe intervient rapidement en cas de sinistre pour minimiser l'impact sur votre bien et accélérer les procédures d'indemnisation.",
                    icon: <Clock className="h-6 w-6 text-[#002395]" />,
                  },
                ].map((item, index) => (
                  <div key={index} className="flex p-4 bg-white rounded-lg shadow-sm border border-gray-100">
                    <div className="mr-4 mt-1">{item.icon}</div>
                    <div>
                      <h3 className="text-lg font-semibold mb-2">{item.title}</h3>
                      <p className="text-gray-700">{item.description}</p>
                    </div>
                  </div>
                ))}
              </div>
            </section>

            {/* Section FAQ */}
            <section className="mb-12" id="faq">
              <h2 className="text-3xl font-bold mb-6 text-[#002395]">
                Questions fréquentes sur l'Assurance Propriétaire Non Occupant
              </h2>
              <div className="space-y-6">
                {[
                  {
                    question: "L'assurance PNO est-elle obligatoire ?",
                    answer:
                      "Oui, l'assurance Propriétaire Non Occupant est légalement obligatoire pour tout propriétaire qui met son bien en location. L'article 7 de la loi n°89-462 du 6 juillet 1989 impose au locataire de s'assurer, mais le propriétaire reste responsable des dommages causés par son bien (vices de construction, défaut d'entretien). De plus, les copropriétés exigent généralement que chaque propriétaire soit assuré, même pour les lots vacants.",
                  },
                  {
                    question: "Quelle est la différence entre l'assurance PNO et l'assurance habitation du locataire ?",
                    answer:
                      "L'assurance habitation du locataire couvre ses biens personnels et sa responsabilité civile en tant qu'occupant. L'assurance PNO, quant à elle, protège le propriétaire pour les dommages causés par le bâtiment lui-même (structure, murs, toiture) et sa responsabilité civile en tant que propriétaire. Les deux assurances sont complémentaires et ne couvrent pas les mêmes risques.",
                  },
                  {
                    question: "Quel est le coût moyen d'une assurance PNO ?",
                    answer:
                      "Le coût d'une assurance PNO varie généralement entre 80€ et 250€ par an, selon la surface du bien, sa localisation, son type (appartement ou maison) et les garanties choisies. Chez DOGES, nous négocions les meilleurs tarifs auprès de nos partenaires assureurs pour vous offrir un rapport qualité-prix optimal.",
                  },
                  {
                    question: "L'assurance PNO couvre-t-elle les loyers impayés ?",
                    answer:
                      "Non, l'assurance PNO ne couvre pas les loyers impayés. Elle protège uniquement contre les dommages matériels et la responsabilité civile du propriétaire. Pour vous protéger contre les impayés, nous proposons une Garantie Loyers Impayés (GLI) complémentaire qui peut être souscrite en parallèle de votre assurance PNO.",
                  },
                  {
                    question: "Que se passe-t-il si je ne souscris pas d'assurance PNO ?",
                    answer:
                      "Sans assurance PNO, vous vous exposez à plusieurs risques : en cas de sinistre, vous devrez assumer seul les frais de réparation de votre bien. De plus, votre responsabilité civile pourrait être engagée si votre bien cause des dommages à des tiers. Enfin, vous pourriez être en infraction avec le règlement de copropriété qui exige généralement une assurance pour chaque lot.",
                  },
                ].map((item, index) => (
                  <div key={index} className="bg-white rounded-lg shadow-sm p-6 border border-gray-100">
                    <h3 className="text-lg font-semibold mb-2">{item.question}</h3>
                    <p className="text-gray-700">{item.answer}</p>
                  </div>
                ))}
              </div>
            </section>

            {/* Section Témoignages */}
            <section className="mb-12" id="temoignages">
              <h2 className="text-3xl font-bold mb-6 text-[#002395]">Ce que disent nos clients</h2>
              <div className="grid md:grid-cols-2 gap-6">
                {[
                  {
                    name: "Philippe R.",
                    location: "Propriétaire à Paris 15ème",
                    testimonial:
                      "Suite à un dégât des eaux important dans mon appartement entre deux locations, l'assurance PNO de DOGES a pris en charge l'intégralité des réparations. Leur réactivité m'a permis de remettre rapidement le bien en location sans perte de revenus.",
                    rating: 5,
                  },
                  {
                    name: "Isabelle M.",
                    location: "Propriétaire à Neuilly-sur-Seine",
                    testimonial:
                      "Je gère plusieurs biens locatifs et j'ai regroupé toutes mes assurances PNO chez DOGES. Leur offre est non seulement compétitive, mais leur service client est exemplaire. Un conseiller dédié répond à toutes mes questions.",
                    rating: 5,
                  },
                  {
                    name: "Laurent D.",
                    location: "Propriétaire à Boulogne-Billancourt",
                    testimonial:
                      "Lorsque mon locataire a accidentellement causé un incendie dans la cuisine, l'assurance PNO a parfaitement complété son assurance habitation pour couvrir les dommages structurels. DOGES a coordonné toutes les démarches entre les assureurs.",
                    rating: 4,
                  },
                  {
                    name: "Catherine B.",
                    location: "Propriétaire à Levallois-Perret",
                    testimonial:
                      "J'ignorais que l'assurance PNO était obligatoire jusqu'à ce que DOGES m'en informe. Leur équipe m'a guidée dans le choix des garanties adaptées à mon studio et m'a même aidée à optimiser mes autres contrats d'assurance.",
                    rating: 5,
                  },
                ].map((testimonial, index) => (
                  <div key={index} className="bg-white rounded-lg shadow-sm p-6 border border-gray-100">
                    <div className="flex items-center mb-4">
                      {[...Array(testimonial.rating)].map((_, i) => (
                        <svg
                          key={i}
                          className="w-5 h-5 text-yellow-400"
                          fill="currentColor"
                          viewBox="0 0 20 20"
                          xmlns="http://www.w3.org/2000/svg"
                        >
                          <path d="M9.049 2.927c.3-.921 1.603-.921 1.902 0l1.07 3.292a1 1 0 00.95.69h3.462c.969 0 1.371 1.24.588 1.81l-2.8 2.034a1 1 0 00-.364 1.118l1.07 3.292c.3.921-.755 1.688-1.54 1.118l-2.8-2.034a1 1 0 00-1.175 0l-2.8 2.034c-.784.57-1.838-.197-1.539-1.118l1.07-3.292a1 1 0 00-.364-1.118L2.98 8.72c-.783-.57-.38-1.81.588-1.81h3.461a1 1 0 00.951-.69l1.07-3.292z"></path>
                        </svg>
                      ))}
                    </div>
                    <p className="text-gray-700 mb-4 italic">"{testimonial.testimonial}"</p>
                    <div>
                      <p className="font-semibold">{testimonial.name}</p>
                      <p className="text-sm text-gray-500">{testimonial.location}</p>
                    </div>
                  </div>
                ))}
              </div>
            </section>

            {/* Formulaire de contact */}
            <section id="contact">
              <h2 className="text-3xl font-bold mb-6 text-[#002395]">
                Demandez un devis pour votre Assurance Propriétaire Non Occupant
              </h2>
              <div className="bg-white rounded-xl shadow-sm p-6 border border-gray-100">
                <SmartContactForm />
              </div>
            </section>
          </div>

          {/* Sidebar */}
          <div className="space-y-8">
            {/* Widget Table des matières */}
            <TableOfContentsWidget
              items={[
                { id: "presentation", title: "Présentation" },
                { id: "avantages", title: "Avantages" },
                { id: "couverture", title: "Couverture" },
                { id: "fonctionnement", title: "Fonctionnement" },
                { id: "pourquoi-nous", title: "Pourquoi nous choisir" },
                { id: "faq", title: "FAQ" },
                { id: "temoignages", title: "Témoignages" },
                { id: "contact", title: "Contact" },
              ]}
            />

            {/* Widget Contact */}
            <ContactWidget/>

            {/* Widget Services associés */}
            <RelatedServicesWidget
              title="Services associés"
              services={[
                {
                  title: "Gestion Locative",
                  description: "Confiez-nous la gestion complète de votre bien immobilier",
                  link: "/services/gestion-locative",
                },
                {
                  title: "Garantie Loyers Impayés",
                  description: "Protégez vos revenus locatifs contre les impayés",
                  link: "/services/garantie-loyers-impayes",
                },
                {
                  title: "Service Juridique",
                  description: "Bénéficiez de notre expertise juridique pour sécuriser votre investissement",
                  link: "/services/juridique",
                },
              ]}
            />
          </div>
        </div>
      </div>
    </>
  )
}
