import { Check, ArrowRight, Building2 } from "lucide-react"
import { Button } from "@/components/ui/button"
import Link from "next/link"
import Image from "next/image"
import type { Metadata } from "next"
import ManagementFeeCalculator from "@/components/management-fee-calculator"
import TableOfContentsWidget from "@/components/table-of-contents-widget"
import RelatedServicesWidget from "@/components/related-services-widget"
import ContactWidget from "@/components/contact-widget"
import PageHeader from "@/components/page-header"

// Métadonnées SEO optimisées
export const metadata: Metadata = {
  title: "Gestion Immobilière Professionnelle | DOGES Immobilier",
  description:
    "Services de gestion immobilière professionnelle pour propriétaires et investisseurs. Maximisez la rentabilité de votre patrimoine immobilier avec DOGES.",
  keywords:
    "gestion immobilière professionnelle, gestion patrimoine immobilier, gestion locative professionnelle, administrateur de biens",
}

export default function GestionImmobiliereProfessionnelle() {
  // Éléments pour la table des matières
  const tocItems = [
    { id: "presentation", title: "Gestion immobilière professionnelle sur mesure" },
    { id: "services", title: "Nos services de gestion immobilière professionnelle" },
    { id: "avantages", title: "Pourquoi choisir notre gestion immobilière professionnelle ?" },
    { id: "calculateur", title: "Estimez vos frais de gestion et prenez rendez-vous" },
    { id: "faq", title: "Questions fréquentes sur la gestion immobilière professionnelle" },
  ]

  // Services associés pour le widget
  const relatedServices = [
    {
      title: "Gestion Locative",
      description: "Gestion complète de vos biens locatifs",
      link: "/services/gestion-locative",
    },
    {
      title: "Administration de Biens",
      description: "Services d'administration immobilière",
      link: "/services/administrateur-biens-paris",
    },
    {
      title: "Garantie Loyers Impayés",
      description: "Protection contre les impayés",
      link: "/services/garantie-loyers-impayes",
    },
  ]

  return (
    <>
      {/* Utilisation du composant PageHeader pour un header standardisé */}
      <PageHeader
        title="Gestion Immobilière Professionnelle"
        description="Maximisez la rentabilité de votre patrimoine immobilier avec notre expertise"
        breadcrumbs={[
          { label: "Accueil", href: "/" },
          { label: "Services", href: "/services" },
          { label: "Gestion Immobilière Professionnelle", href: "/services/gestion-immobiliere-professionnelle" },
        ]}
        backgroundImage="https://images.unsplash.com/photo-1560520653-9e0e4c89eb11?q=80&w=2073&auto=format&fit=crop"
      />

      {/* Contenu principal avec sidebar */}
      <div className="container mx-auto px-4 py-12">
        <div className="flex flex-col lg:flex-row gap-8">
          {/* Contenu principal (2/3) */}
          <div className="lg:w-2/3">
            {/* Section de présentation */}
            <section className="mb-16" id="presentation">
              <div className="grid lg:grid-cols-2 gap-12 items-center">
                <div>
                  <h2 className="text-3xl font-bold mb-6 text-[#002395]">
                    Gestion immobilière professionnelle sur mesure
                  </h2>
                  <p className="text-lg text-gray-700 mb-6">
                    DOGES vous propose une <strong>gestion immobilière professionnelle</strong> complète et
                    personnalisée pour valoriser votre patrimoine et optimiser sa rentabilité. Notre équipe d'experts
                    vous accompagne dans tous les aspects de la gestion de vos biens immobiliers.
                  </p>
                  <p className="text-lg text-gray-700 mb-6">
                    Que vous soyez propriétaire d'un seul bien ou d'un portefeuille immobilier conséquent, nous vous
                    offrons des solutions adaptées à vos besoins spécifiques et à vos objectifs patrimoniaux.
                  </p>
                  <div className="space-y-4">
                    {[
                      "Gestion locative complète",
                      "Suivi technique et maintenance",
                      "Gestion administrative et comptable",
                      "Optimisation fiscale et patrimoniale",
                      "Reporting régulier et transparent",
                    ].map((feature, index) => (
                      <div key={index} className="flex items-start">
                        <div className="mt-1 mr-3 bg-[#002395] rounded-full p-1">
                          <Check className="h-4 w-4 text-white" />
                        </div>
                        <p>{feature}</p>
                      </div>
                    ))}
                  </div>
                </div>
                <div className="relative h-[500px] rounded-2xl overflow-hidden">
                  <Image
                    src="https://images.unsplash.com/photo-1486406146926-c627a92ad1ab?q=80&w=2070&auto=format&fit=crop"
                    alt="Gestion immobilière professionnelle"
                    fill
                    className="object-cover"
                  />
                </div>
              </div>
            </section>

            {/* Nos services de gestion immobilière */}
            <section className="py-16 -mx-8 px-8 bg-gray-50" id="services">
              <h2 className="text-3xl font-bold mb-12 text-center text-[#002395]">
                Nos services de gestion immobilière professionnelle
              </h2>
              <div className="grid md:grid-cols-3 gap-8">
                {[
                  {
                    title: "Gestion locative",
                    description:
                      "Recherche et sélection de locataires, rédaction des baux, états des lieux, encaissement des loyers, gestion des charges et des travaux.",
                    icon: <Building2 className="h-10 w-10 text-white" />,
                    link: "/services/gestion-locative",
                  },
                  {
                    title: "Gestion technique",
                    description:
                      "Suivi de l'entretien courant, coordination des travaux, interventions d'urgence, mise en conformité technique et réglementaire.",
                    icon: <Building2 className="h-10 w-10 text-white" />,
                    link: "/services/gestion-technique",
                  },
                  {
                    title: "Gestion administrative et comptable",
                    description:
                      "Suivi des paiements, régularisation des charges, déclarations fiscales, reporting financier détaillé et personnalisé.",
                    icon: <Building2 className="h-10 w-10 text-white" />,
                    link: "/services/gestion-administrative",
                  },
                ].map((service, index) => (
                  <div key={index} className="bg-white rounded-xl shadow-lg p-8 border border-gray-100">
                    <div className="bg-gradient-to-r from-[#002395] to-[#001875] w-16 h-16 rounded-full flex items-center justify-center mb-6">
                      {service.icon}
                    </div>
                    <h3 className="text-xl font-bold mb-4">{service.title}</h3>
                    <p className="text-gray-700 mb-6">{service.description}</p>
                    <Link href={service.link}>
                      <Button
                        variant="outline"
                        className="border-[#002395] text-[#002395] hover:bg-[#002395] hover:text-white"
                      >
                        En savoir plus
                        <ArrowRight className="ml-2 h-4 w-4" />
                      </Button>
                    </Link>
                  </div>
                ))}
              </div>
            </section>

            {/* Pourquoi choisir notre gestion immobilière */}
            <section className="py-16" id="avantages">
              <div className="grid lg:grid-cols-2 gap-12 items-center">
                <div className="order-2 lg:order-1">
                  <h2 className="text-3xl font-bold mb-6 text-[#002395]">
                    Pourquoi choisir notre gestion immobilière professionnelle ?
                  </h2>
                  <div className="space-y-6">
                    {[
                      {
                        title: "Expertise et expérience",
                        description:
                          "Plus de 25 ans d'expérience dans la gestion immobilière et une connaissance approfondie du marché.",
                      },
                      {
                        title: "Approche personnalisée",
                        description:
                          "Solutions sur mesure adaptées à vos besoins spécifiques et à vos objectifs patrimoniaux.",
                      },
                      {
                        title: "Transparence totale",
                        description:
                          "Reporting régulier et accès à votre espace client en ligne pour suivre la gestion de vos biens.",
                      },
                      {
                        title: "Optimisation de la rentabilité",
                        description:
                          "Stratégies pour maximiser vos revenus locatifs et valoriser votre patrimoine immobilier.",
                      },
                      {
                        title: "Réseau de professionnels",
                        description:
                          "Collaboration avec un réseau d'artisans et de prestataires qualifiés pour l'entretien de vos biens.",
                      },
                    ].map((advantage, index) => (
                      <div key={index} className="flex items-start">
                        <div className="mt-1 mr-3 bg-[#002395] rounded-full p-1">
                          <Check className="h-4 w-4 text-white" />
                        </div>
                        <div>
                          <h3 className="font-semibold">{advantage.title}</h3>
                          <p className="text-gray-700">{advantage.description}</p>
                        </div>
                      </div>
                    ))}
                  </div>
                </div>
                <div className="order-1 lg:order-2 relative h-[400px] rounded-2xl overflow-hidden">
                  <Image
                    src="https://images.unsplash.com/photo-1450101499163-c8848c66ca85?q=80&w=2070&auto=format&fit=crop"
                    alt="Gestion immobilière professionnelle"
                    fill
                    className="object-cover"
                  />
                </div>
              </div>
            </section>

            {/* Calculateur et prise de rendez-vous */}
            <section className="py-16 -mx-8 px-8 bg-gray-50" id="calculateur">
              <h2 className="text-3xl font-bold mb-12 text-center text-[#002395]">
                Estimez vos frais de gestion et prenez rendez-vous
              </h2>
              <div className="grid md:grid-cols-2 gap-8">
                <div>
                  <ManagementFeeCalculator />
                </div>
                <div className="bg-white rounded-xl shadow-lg p-8 border border-gray-100">
                  <h3 className="text-xl font-semibold mb-6">Prenez rendez-vous avec un expert</h3>
                  <p className="text-gray-700 mb-6">
                    Vous souhaitez en savoir plus sur nos services de gestion immobilière professionnelle ? Prenez
                    rendez-vous avec l'un de nos experts pour discuter de vos besoins spécifiques.
                  </p>
                  <button className="w-full bg-[#002395] text-white py-2 px-4 rounded hover:bg-[#001875] transition-colors">
                    Prendre rendez-vous
                  </button>
                </div>
              </div>
            </section>

            {/* FAQ pour le SEO */}
            <section className="py-16" id="faq">
              <h2 className="text-3xl font-bold mb-12 text-center text-[#002395]">
                Questions fréquentes sur la gestion immobilière professionnelle
              </h2>
              <div className="space-y-6">
                {[
                  {
                    question: "Quels sont les avantages de confier la gestion de mon bien à un professionnel ?",
                    answer:
                      "Confier la gestion de votre bien à un professionnel comme DOGES vous fait gagner du temps et de l'argent. Vous bénéficiez de notre expertise pour la sélection des locataires, la gestion administrative et technique, et l'optimisation fiscale. Nous assurons également une veille juridique constante pour vous garantir la conformité avec les évolutions législatives.",
                  },
                  {
                    question: "Comment sont calculés les honoraires de gestion immobilière ?",
                    answer:
                      "Les honoraires de gestion immobilière sont généralement calculés en pourcentage des loyers encaissés, entre 6% et 10% HT selon le niveau de service et la complexité de la gestion. Chez DOGES, nous proposons différentes formules adaptées à vos besoins, avec une transparence totale sur les tarifs.",
                  },
                  {
                    question: "Quelle est la différence entre un gestionnaire immobilier et un syndic de copropriété ?",
                    answer:
                      "Un gestionnaire immobilier s'occupe de la gestion locative d'un bien pour le compte d'un propriétaire (recherche de locataires, encaissement des loyers, etc.), tandis qu'un syndic de copropriété gère les parties communes d'un immeuble pour le compte de l'ensemble des copropriétaires. DOGES propose ces deux services complémentaires.",
                  },
                ].map((item, index) => (
                  <div key={index} className="bg-white rounded-lg shadow-sm p-6 border border-gray-100">
                    <h3 className="text-lg font-semibold mb-2">{item.question}</h3>
                    <p className="text-gray-700">{item.answer}</p>
                  </div>
                ))}
              </div>
            </section>
          </div>

          {/* Sidebar (1/3) */}
          <div className="lg:w-1/3 space-y-8">
            {/* Table des matières */}
            <div className="sticky top-24">
              <TableOfContentsWidget items={tocItems} />

              {/* Services associés */}
              <div className="mt-8">
                <RelatedServicesWidget services={relatedServices} />
              </div>

              {/* Contact rapide */}
              <div className="mt-8">
                <ContactWidget />
              </div>
            </div>
          </div>
        </div>
      </div>

      {/* CTA */}
      <section className="py-16 bg-gradient-to-r from-[#002395] to-[#001875] text-white">
        <div className="container mx-auto px-4 text-center">
          <h2 className="text-3xl font-bold mb-6">Confiez la gestion de votre patrimoine immobilier à DOGES</h2>
          <p className="text-xl mb-8 max-w-2xl mx-auto">
            Bénéficiez de notre expertise et de notre engagement pour une gestion optimale de vos biens immobiliers.
          </p>
          <div className="flex flex-col sm:flex-row justify-center gap-4">
            <button className="bg-white text-[#002395] py-2 px-6 rounded hover:bg-gray-100 transition-colors">
              Prendre rendez-vous
            </button>
            <button className="bg-transparent border-2 border-white text-white py-2 px-6 rounded hover:bg-white hover:text-[#002395] transition-colors">
              Demander un devis
            </button>
          </div>
        </div>
      </section>
    </>
  )
}
