import type { Metadata } from "next"
import Image from "next/image"
import Link from "next/link"
import { Check, ArrowRight, Scale, FileText, Shield, Gavel, BookOpen } from "lucide-react"
import { Card, CardContent } from "@/components/ui/card"
import ServiceHeader from "@/components/service-header"
import RelatedServicesWidget from "@/components/related-services-widget"
import TableOfContentsWidget from "@/components/table-of-contents-widget"
import { ConsultationCTA } from "@/components/consultation-cta"

export const metadata: Metadata = {
  title: "Service Juridique Immobilier à Paris | DOGES Immobilier",
  description:
    "Services juridiques immobiliers à Paris. Conseil, rédaction de contrats, gestion des contentieux et assistance juridique avec DOGES, votre partenaire de confiance.",
  keywords:
    "service juridique immobilier, conseil juridique immobilier, contentieux immobilier, droit immobilier Paris",
}

export default function JuridiquePage() {
  // Configuration de la table des matières
  const tocItems = [
    { id: "presentation", title: "Notre expertise juridique" },
    { id: "services", title: "Nos services juridiques" },
    { id: "avantages", title: "Pourquoi nous choisir" },
    { id: "processus", title: "Notre processus" },
    { id: "temoignages", title: "Témoignages clients" },
    { id: "faq", title: "Questions fréquentes" },
  ]

  // Services associés pertinents
  const relatedServices = [
    {
      title: "Gestion Locative",
      description: "Confiez-nous la gestion complète de votre bien immobilier",
      link: "/services/gestion-locative",
    },
    {
      title: "Transaction Immobilière",
      description: "Achat, vente et estimation de biens immobiliers",
      link: "/services/transaction",
    },
    {
      title: "Garantie Loyers Impayés",
      description: "Protégez vos revenus locatifs contre les impayés",
      link: "/services/garantie-loyers-impayes",
    },
  ]

  return (
    <>
      {/* En-tête de service avec image de fond */}
      <ServiceHeader
        title="Service Juridique Immobilier"
        description="Conseil, rédaction de contrats et gestion des contentieux immobiliers à Paris et en Île-de-France"
        breadcrumbItems={[
          { name: "Accueil", url: "/" },
          { name: "Services", url: "/services" },
          { name: "Service Juridique", url: "/services/juridique" },
        ]}
        backgroundImage="/legal-real-estate-professional-contract.png"
      />

      <div className="container mx-auto px-4 py-12">
        <div className="flex flex-col lg:flex-row gap-8">
          {/* Contenu principal */}
          <div className="lg:w-2/3">
            {/* Présentation */}
            <section className="mb-16" id="presentation">
              <h2 className="text-3xl font-bold mb-6 text-[#002395]">Notre expertise juridique immobilière</h2>
              <div className="grid lg:grid-cols-2 gap-12 items-center">
                <div>
                  <p className="text-lg text-gray-700 mb-6">
                    Chez DOGES, notre service juridique immobilier vous accompagne dans toutes vos démarches et
                    problématiques juridiques liées à l'immobilier à Paris et en Île-de-France. Notre équipe d'experts
                    en droit immobilier vous offre un service personnalisé pour sécuriser vos opérations et défendre vos
                    intérêts.
                  </p>
                  <p className="text-lg text-gray-700 mb-6">
                    Que vous soyez propriétaire, investisseur, locataire ou copropriétaire, notre équipe juridique vous
                    propose un accompagnement sur mesure et des solutions adaptées à vos besoins spécifiques.
                  </p>
                  <div className="space-y-4">
                    {[
                      "Conseil juridique personnalisé",
                      "Rédaction et analyse de contrats",
                      "Gestion des contentieux locatifs",
                      "Assistance en copropriété",
                      "Veille juridique et réglementaire",
                    ].map((feature, index) => (
                      <div key={index} className="flex items-start">
                        <div className="mt-1 mr-3 bg-[#002395] rounded-full p-1">
                          <Check className="h-4 w-4 text-white" />
                        </div>
                        <p>{feature}</p>
                      </div>
                    ))}
                  </div>
                </div>
                <div className="relative h-[400px] rounded-2xl overflow-hidden">
                  <Image
                    src="/legal-real-estate-professional-contract.png"
                    alt="Conseiller juridique immobilier"
                    fill
                    className="object-cover"
                  />
                </div>
              </div>
            </section>

            {/* Nos services juridiques */}
            <section className="mb-16 bg-gray-50 -mx-4 px-4 py-12" id="services">
              <h2 className="text-3xl font-bold mb-12 text-center text-[#002395]">
                Nos services juridiques immobiliers
              </h2>

              {/* Tableau des services juridiques */}
              <div className="overflow-x-auto mb-8">
                <table className="w-full bg-white rounded-xl shadow-sm border-collapse">
                  <thead>
                    <tr className="bg-[#002395] text-white">
                      <th className="py-4 px-6 text-left rounded-tl-xl">Service</th>
                      <th className="py-4 px-6 text-left">Description</th>
                      <th className="py-4 px-6 text-left rounded-tr-xl">Avantages</th>
                    </tr>
                  </thead>
                  <tbody>
                    <tr className="border-b border-gray-200">
                      <td className="py-4 px-6 font-medium">Conseil juridique</td>
                      <td className="py-4 px-6">
                        Consultation et conseil personnalisé sur toutes les questions juridiques liées à l'immobilier.
                      </td>
                      <td className="py-4 px-6">
                        <ul className="list-disc pl-5 space-y-1">
                          <li>Expertise en droit immobilier</li>
                          <li>Réponses adaptées à votre situation</li>
                          <li>Prévention des risques juridiques</li>
                        </ul>
                      </td>
                    </tr>
                    <tr className="border-b border-gray-200">
                      <td className="py-4 px-6 font-medium">Rédaction de contrats</td>
                      <td className="py-4 px-6">
                        Élaboration et révision de contrats de bail, promesses de vente, et autres documents juridiques.
                      </td>
                      <td className="py-4 px-6">
                        <ul className="list-disc pl-5 space-y-1">
                          <li>Contrats sécurisés et conformes</li>
                          <li>Protection de vos intérêts</li>
                          <li>Adaptation aux spécificités de votre situation</li>
                        </ul>
                      </td>
                    </tr>
                    <tr>
                      <td className="py-4 px-6 font-medium rounded-bl-xl">Gestion des contentieux</td>
                      <td className="py-4 px-6">
                        Accompagnement dans la résolution des litiges locatifs, de copropriété ou liés aux transactions.
                      </td>
                      <td className="py-4 px-6 rounded-br-xl">
                        <ul className="list-disc pl-5 space-y-1">
                          <li>Stratégie de résolution efficace</li>
                          <li>Représentation devant les instances</li>
                          <li>Suivi personnalisé de votre dossier</li>
                        </ul>
                      </td>
                    </tr>
                  </tbody>
                </table>
              </div>

              {/* Cartes de services avec plus de détails */}
              <div className="grid md:grid-cols-3 gap-8">
                {[
                  {
                    title: "Conseil juridique",
                    icon: <Scale className="h-10 w-10 text-[#002395] mb-4" />,
                    description:
                      "Notre service de conseil juridique vous apporte des réponses précises et adaptées à toutes vos questions en matière de droit immobilier.",
                    details: [
                      "Analyse de votre situation juridique",
                      "Conseil sur les aspects réglementaires",
                      "Orientation stratégique",
                      "Veille juridique personnalisée",
                      "Prévention des risques",
                    ],
                  },
                  {
                    title: "Rédaction de contrats",
                    icon: <FileText className="h-10 w-10 text-[#002395] mb-4" />,
                    description:
                      "Notre service de rédaction de contrats vous garantit des documents juridiques sécurisés et conformes à la législation en vigueur.",
                    details: [
                      "Contrats de bail résidentiel et commercial",
                      "Promesses de vente et compromis",
                      "Mandats de gestion",
                      "Règlements de copropriété",
                      "Révision et analyse de contrats existants",
                    ],
                  },
                  {
                    title: "Gestion des contentieux",
                    icon: <Gavel className="h-10 w-10 text-[#002395] mb-4" />,
                    description:
                      "Notre service de gestion des contentieux vous accompagne dans la résolution de vos litiges immobiliers avec efficacité et professionnalisme.",
                    details: [
                      "Impayés de loyers et charges",
                      "Litiges de copropriété",
                      "Contentieux liés aux travaux",
                      "Procédures d'expulsion",
                      "Médiation et négociation",
                    ],
                  },
                ].map((service, index) => (
                  <Card key={index} className="bg-white rounded-xl shadow-lg border border-gray-100">
                    <CardContent className="p-6">
                      <div className="flex justify-center">{service.icon}</div>
                      <h3 className="text-xl font-bold mb-4 text-center">{service.title}</h3>
                      <p className="text-gray-700 mb-4">{service.description}</p>
                      <div className="space-y-2">
                        {service.details.map((detail, idx) => (
                          <div key={idx} className="flex items-start">
                            <div className="mt-1 mr-2 bg-[#002395] rounded-full p-0.5">
                              <Check className="h-3 w-3 text-white" />
                            </div>
                            <p className="text-sm">{detail}</p>
                          </div>
                        ))}
                      </div>
                    </CardContent>
                  </Card>
                ))}
              </div>
            </section>

            {/* Pourquoi nous choisir */}
            <section className="mb-16" id="avantages">
              <h2 className="text-3xl font-bold mb-6 text-[#002395]">Pourquoi choisir notre service juridique</h2>
              <div className="grid lg:grid-cols-2 gap-12 items-center">
                <div className="order-2 lg:order-1">
                  <div className="space-y-6">
                    {[
                      {
                        title: "Expertise spécialisée",
                        description:
                          "Une équipe d'experts en droit immobilier avec une connaissance approfondie de la législation et de la jurisprudence.",
                      },
                      {
                        title: "Approche préventive",
                        description:
                          "Nous privilégions la prévention des litiges par un conseil juridique proactif et une rédaction rigoureuse des contrats.",
                      },
                      {
                        title: "Réactivité et disponibilité",
                        description: "Une réponse rapide à vos questions et un suivi régulier de vos dossiers.",
                      },
                      {
                        title: "Solutions sur mesure",
                        description:
                          "Des conseils et des stratégies adaptés à votre situation spécifique et à vos objectifs.",
                      },
                      {
                        title: "Veille juridique constante",
                        description:
                          "Une mise à jour permanente de nos connaissances pour vous garantir un conseil conforme aux dernières évolutions législatives.",
                      },
                    ].map((advantage, index) => (
                      <div key={index} className="flex items-start">
                        <div className="mt-1 mr-3 bg-[#002395] rounded-full p-1">
                          <Check className="h-4 w-4 text-white" />
                        </div>
                        <div>
                          <h3 className="font-semibold">{advantage.title}</h3>
                          <p className="text-gray-700">{advantage.description}</p>
                        </div>
                      </div>
                    ))}
                  </div>
                </div>
                <div className="order-1 lg:order-2 relative h-[400px] rounded-2xl overflow-hidden">
                  <Image
                    src="/legal-real-estate-professional-contract.png"
                    alt="Documents juridiques immobiliers"
                    fill
                    className="object-cover"
                  />
                </div>
              </div>
            </section>

            {/* Notre processus */}
            <section className="mb-16" id="processus">
              <h2 className="text-3xl font-bold mb-6 text-[#002395]">Notre processus d'accompagnement juridique</h2>
              <div className="space-y-8">
                {[
                  {
                    step: "1",
                    title: "Analyse de votre situation",
                    description:
                      "Nous commençons par une analyse approfondie de votre situation juridique et de vos besoins spécifiques pour identifier les enjeux et les risques.",
                  },
                  {
                    step: "2",
                    title: "Conseil et stratégie",
                    description:
                      "Nous vous proposons un conseil personnalisé et élaborons une stratégie juridique adaptée à vos objectifs et à votre situation.",
                  },
                  {
                    step: "3",
                    title: "Mise en œuvre",
                    description:
                      "Nous mettons en œuvre la stratégie définie, que ce soit par la rédaction de documents juridiques, la négociation ou la représentation dans un contentieux.",
                  },
                  {
                    step: "4",
                    title: "Suivi et ajustement",
                    description:
                      "Nous assurons un suivi régulier de votre dossier et adaptons notre stratégie en fonction de l'évolution de votre situation et des procédures en cours.",
                  },
                  {
                    step: "5",
                    title: "Résolution et bilan",
                    description:
                      "Nous finalisons la résolution de votre problématique juridique et établissons un bilan pour prévenir d'éventuelles difficultés futures.",
                  },
                ].map((step, index) => (
                  <div key={index} className="flex gap-6">
                    <div className="flex-shrink-0 w-12 h-12 rounded-full bg-[#002395] text-white flex items-center justify-center text-xl font-bold">
                      {step.step}
                    </div>
                    <div>
                      <h3 className="text-xl font-semibold mb-2">{step.title}</h3>
                      <p className="text-gray-700">{step.description}</p>
                    </div>
                  </div>
                ))}
              </div>
            </section>

            {/* Témoignages */}
            <section className="mb-16" id="temoignages">
              <h2 className="text-3xl font-bold mb-6 text-[#002395]">Ce que nos clients disent de nous</h2>
              <div className="grid md:grid-cols-2 gap-6">
                {[
                  {
                    name: "Laurent D.",
                    role: "Propriétaire à Paris 8e",
                    testimonial:
                      "L'équipe juridique de DOGES m'a aidé à résoudre un contentieux locatif complexe avec professionnalisme. Leur expertise et leur réactivité ont été déterminantes dans la résolution rapide de mon problème.",
                  },
                  {
                    name: "Isabelle M.",
                    role: "Investisseur immobilier",
                    testimonial:
                      "Je fais appel aux services juridiques de DOGES pour tous mes investissements. Leurs conseils avisés et la qualité de leurs contrats me permettent d'investir en toute sécurité.",
                  },
                  {
                    name: "Syndicat de copropriété",
                    role: "Résidence Les Ormes, Paris 16e",
                    testimonial:
                      "DOGES nous accompagne dans la gestion juridique de notre copropriété depuis 3 ans. Leur expertise nous a permis de résoudre plusieurs litiges et d'optimiser notre règlement de copropriété.",
                  },
                  {
                    name: "Philippe B.",
                    role: "Bailleur multi-lots",
                    testimonial:
                      "Grâce au service juridique de DOGES, j'ai pu récupérer des loyers impayés et mettre en place des procédures efficaces pour sécuriser mes revenus locatifs. Un accompagnement de qualité que je recommande.",
                  },
                ].map((testimonial, index) => (
                  <div key={index} className="bg-white p-6 rounded-xl shadow-sm border border-gray-100">
                    <div className="flex items-center mb-4">
                      <div className="w-12 h-12 rounded-full bg-gray-200 flex items-center justify-center text-gray-500 font-bold">
                        {testimonial.name.charAt(0)}
                      </div>
                      <div className="ml-4">
                        <h4 className="font-semibold">{testimonial.name}</h4>
                        <p className="text-sm text-gray-600">{testimonial.role}</p>
                      </div>
                    </div>
                    <p className="italic text-gray-700">"{testimonial.testimonial}"</p>
                  </div>
                ))}
              </div>
            </section>

            {/* FAQ */}
            <section className="mb-16" id="faq">
              <h2 className="text-3xl font-bold mb-6 text-[#002395]">Questions fréquentes</h2>
              <div className="space-y-6">
                {[
                  {
                    question: "Quels types de contentieux immobiliers pouvez-vous gérer ?",
                    answer:
                      "Notre service juridique gère tous types de contentieux immobiliers : impayés de loyers, procédures d'expulsion, litiges liés aux travaux, conflits de voisinage, contentieux de copropriété, litiges liés aux transactions immobilières, etc. Notre équipe d'experts vous accompagne de la phase précontentieuse jusqu'à la résolution définitive du litige, en privilégiant lorsque c'est possible les solutions amiables avant d'engager des procédures judiciaires.",
                  },
                  {
                    question: "Comment se déroule une consultation juridique chez DOGES ?",
                    answer:
                      "Une consultation juridique chez DOGES commence par un entretien approfondi pour comprendre votre situation et vos besoins. Notre juriste analyse ensuite les documents pertinents et vous présente une analyse juridique de votre cas avec les options possibles. Vous recevez ensuite un compte-rendu écrit avec nos recommandations. Selon la complexité de votre dossier, nous pouvons vous proposer un accompagnement sur mesure avec un suivi régulier.",
                  },
                  {
                    question: "Quels documents faut-il préparer pour un rendez-vous juridique ?",
                    answer:
                      "Pour optimiser votre rendez-vous juridique, nous vous recommandons de préparer tous les documents relatifs à votre situation : contrats (bail, promesse de vente, etc.), correspondances échangées avec les parties concernées, mises en demeure, procès-verbaux d'assemblée générale pour les questions de copropriété, factures de travaux en cas de litige, et tout autre document pertinent. Plus votre dossier sera complet, plus notre analyse sera précise et efficace.",
                  },
                  {
                    question: "Quels sont vos tarifs pour les services juridiques ?",
                    answer:
                      "Nos tarifs varient selon la nature et la complexité de votre dossier. Nous proposons des forfaits pour les prestations standards (rédaction de bail, consultation simple, etc.) et des honoraires au temps passé pour les dossiers complexes. Dans tous les cas, nous vous présentons un devis détaillé avant toute intervention. Pour nos clients en gestion locative, nous proposons des tarifs préférentiels et certaines prestations juridiques peuvent être incluses dans votre contrat de gestion.",
                  },
                  {
                    question: "Combien de temps faut-il pour résoudre un contentieux locatif ?",
                    answer:
                      "Le délai de résolution d'un contentieux locatif varie considérablement selon la nature du litige et la voie de résolution choisie. Une résolution amiable peut aboutir en quelques semaines, tandis qu'une procédure judiciaire peut s'étendre sur plusieurs mois, voire plus d'un an pour les cas complexes. Notre équipe s'efforce de trouver les solutions les plus rapides et efficaces, en privilégiant lorsque c'est possible les modes alternatifs de résolution des conflits (médiation, conciliation) pour accélérer le processus.",
                  },
                ].map((item, index) => (
                  <div key={index} className="bg-white rounded-lg shadow-sm p-6 border border-gray-100">
                    <h3 className="text-lg font-semibold mb-2">{item.question}</h3>
                    <p className="text-gray-700">{item.answer}</p>
                  </div>
                ))}
              </div>
              <div className="mt-6 text-center">
                <Link href="/faq" className="text-[#002395] font-medium hover:underline inline-flex items-center">
                  Voir toutes les questions fréquentes
                  <ArrowRight className="ml-2 h-4 w-4" />
                </Link>
              </div>
            </section>
          </div>

          {/* Sidebar */}
          <div className="lg:w-1/3">
            {/* Table des matières */}
            <div className="mb-6">
              <TableOfContentsWidget items={tocItems} />
            </div>

            {/* Widget de contact rapide */}
            <div className="mb-6">
              <Card className="overflow-hidden border border-gray-200">
                <CardContent className="p-0">
                  <div className="bg-[#002395] p-4 text-white">
                    <h3 className="text-xl font-bold">Besoin d'un conseil juridique ?</h3>
                    <p className="text-white/80 mt-1">Nos experts sont à votre disposition</p>
                  </div>
                  <div className="p-6">
                    <p className="text-gray-700 mb-4">
                      Prenez rendez-vous avec notre équipe juridique pour une consultation personnalisée.
                    </p>
                    <div className="space-y-3">
                      <div className="flex items-start">
                        <Shield className="h-5 w-5 text-[#002395] mr-2 mt-0.5" />
                        <p className="text-sm text-gray-600">Expertise en droit immobilier</p>
                      </div>
                      <div className="flex items-start">
                        <BookOpen className="h-5 w-5 text-[#002395] mr-2 mt-0.5" />
                        <p className="text-sm text-gray-600">Analyse personnalisée de votre situation</p>
                      </div>
                      <div className="flex items-start">
                        <Gavel className="h-5 w-5 text-[#002395] mr-2 mt-0.5" />
                        <p className="text-sm text-gray-600">Solutions juridiques adaptées</p>
                      </div>
                    </div>
                    <div className="mt-6">
                      <Link href="/contact" className="block w-full">
                        <button className="w-full bg-[#002395] hover:bg-[#001875] text-white py-3 px-4 rounded font-medium transition-colors">
                          Prendre rendez-vous
                        </button>
                      </Link>
                    </div>
                  </div>
                </CardContent>
              </Card>
            </div>

            {/* Widget d'information juridique */}
            <div className="mb-6">
              <Card className="border border-gray-200">
                <CardContent className="p-6">
                  <h3 className="text-xl font-bold mb-4">Actualités juridiques</h3>
                  <div className="space-y-4">
                    <div>
                      <h4 className="font-medium text-[#002395]">Encadrement des loyers</h4>
                      <p className="text-sm text-gray-600 mt-1">
                        Dernières évolutions de la réglementation sur l'encadrement des loyers à Paris.
                      </p>
                    </div>
                    <div>
                      <h4 className="font-medium text-[#002395]">Réforme des diagnostics énergétiques</h4>
                      <p className="text-sm text-gray-600 mt-1">
                        Impact des nouvelles normes DPE sur la location et la vente de biens immobiliers.
                      </p>
                    </div>
                    <div>
                      <h4 className="font-medium text-[#002395]">Contentieux locatifs</h4>
                      <p className="text-sm text-gray-600 mt-1">
                        Évolutions jurisprudentielles récentes en matière d'impayés et d'expulsion.
                      </p>
                    </div>
                  </div>
                  <div className="mt-4 pt-4 border-t border-gray-200">
                    <Link href="/blog" className="text-[#002395] font-medium hover:underline inline-flex items-center">
                      Consulter notre blog juridique
                      <ArrowRight className="ml-2 h-4 w-4" />
                    </Link>
                  </div>
                </CardContent>
              </Card>
            </div>

            {/* Services associés */}
            <RelatedServicesWidget services={relatedServices} />
          </div>
        </div>

        {/* CTA Section */}
        <div className="mt-12">
          <ConsultationCTA
            title="Besoin d'un accompagnement juridique pour votre bien immobilier ?"
            description="Nos experts juridiques vous accompagnent dans toutes vos démarches et problématiques immobilières."
            buttonText="Prendre rendez-vous"
            buttonLink="/contact"
          />
        </div>
      </div>
    </>
  )
}
