import type { Metadata } from "next"
import Image from "next/image"
import { Check, ArrowRight, Building2, Shield, Clock, FileText } from "lucide-react"
import { Button } from "@/components/ui/button"
import Link from "next/link"
import Breadcrumb from "@/components/breadcrumb"
import {TableOfContents} from "@/components/table-of-contents"
import TableOfContentsWidget from "@/components/table-of-contents-widget"
import RelatedServicesWidget from "@/components/related-services-widget"
import ContactWidget from "@/components/contact-widget"

// Métadonnées SEO optimisées
export const metadata: Metadata = {
  title: "Syndic de Copropriété Paris | DOGES Immobilier",
  description:
    "DOGES, votre syndic de copropriété à Paris. Gestion transparente et efficace de votre copropriété parisienne avec un service personnalisé.",
  keywords: "syndic copropriété Paris, syndic professionnel Paris, gestion copropriété Paris, syndic immeuble Paris",
}

export default function SyndicCoproprieteParis() {
  // Définition des sections pour la table des matières
  const tocItems = [
    { id: "presentation", title: "Syndic de copropriété professionnel à Paris" },
    { id: "services", title: "Nos services de syndic de copropriété à Paris" },
    { id: "avantages", title: "Pourquoi choisir DOGES comme syndic de copropriété à Paris ?" },
    { id: "tarifs", title: "Nos tarifs de syndic de copropriété à Paris" },
    { id: "faq", title: "Questions fréquentes sur le syndic de copropriété à Paris" },
  ]

  return (
    <>
      {/* En-tête avec image de fond */}
      <div className="relative">
        {/* Image de fond avec overlay */}
        <div className="absolute inset-0 z-0">
          <Image
            src="https://images.unsplash.com/photo-1486406146926-c627a92ad1ab?q=80&w=2070&auto=format&fit=crop"
            alt="Syndic de copropriété à Paris"
            fill
            className="object-cover"
            priority
          />
          <div className="absolute inset-0 bg-[#002395]/90"></div>
        </div>

        {/* Contenu de l'en-tête */}
        <div className="relative z-10 container mx-auto px-4 py-16 text-white">
          {/* Fil d'Ariane */}
          <Breadcrumb
            items={[
              { name: "Accueil", url: "/" },
              { name: "Services", url: "/services" },
              { name: "Syndic de Copropriété Paris", url: "/services/syndic-copropriete-paris" },
            ]}
            className="mb-12 text-white/80"
          />

          {/* Titre et sous-titre */}
          <div className="max-w-3xl">
            <h1 className="text-4xl md:text-5xl font-bold mb-4">Syndic de Copropriété à Paris</h1>
            <p className="text-xl text-white/90 mb-8">
              DOGES, votre partenaire de confiance pour la gestion transparente et efficace de votre copropriété
              parisienne
            </p>

            {/* Boutons CTA */}
            <div className="flex flex-wrap gap-4">
              <Link href="/contact">
                <Button className="bg-white text-[#002395] hover:bg-white/90 hover:text-[#002395] px-6 py-3 rounded font-medium transition-colors">
                  Demander un devis gratuit
                </Button>
              </Link>
              <Link href="#presentation">
                <Button className="bg-transparent text-white border-2 border-white hover:bg-white hover:text-[#002395] px-6 py-3 rounded font-medium transition-colors">
                  En savoir plus
                </Button>
              </Link>
            </div>
          </div>
        </div>
      </div>

      {/* Contenu principal */}
      <div className="container mx-auto px-4 py-12">
        <div className="grid lg:grid-cols-3 gap-12">
          {/* Colonne principale (2/3) */}
          <div className="lg:col-span-2">
            {/* Table des matières pour mobile */}
            <div className="lg:hidden mb-8">
              <TableOfContents
                items={[
                  { id: "presentation", text: "Syndic de copropriété professionnel à Paris", level: 2 },
                  { id: "services", text: "Nos services de syndic de copropriété à Paris", level: 2 },
                  { id: "avantages", text: "Pourquoi choisir DOGES comme syndic de copropriété à Paris ?", level: 2 },
                  { id: "tarifs", text: "Nos tarifs de syndic de copropriété à Paris", level: 2 },
                  { id: "faq", text: "Questions fréquentes sur le syndic de copropriété à Paris", level: 2 },
                ]}
              />
            </div>

            {/* Section de présentation */}
            <section className="mb-12" id="presentation">
              <h2 className="text-3xl font-bold mb-6 text-[#002395]">Syndic de copropriété professionnel à Paris</h2>
              <div className="grid lg:grid-cols-2 gap-12 items-center">
                <div>
                  <p className="text-lg text-gray-700 mb-6">
                    DOGES est un <strong>syndic de copropriété professionnel à Paris</strong> qui s'engage à offrir une
                    gestion transparente, réactive et efficace de votre immeuble. Notre équipe expérimentée connaît
                    parfaitement les spécificités des copropriétés parisiennes et les défis qu'elles peuvent rencontrer.
                  </p>
                  <p className="text-lg text-gray-700 mb-6">
                    Nous mettons un point d'honneur à établir une relation de confiance avec les copropriétaires et les
                    conseils syndicaux, en privilégiant la communication et la transparence dans toutes nos actions.
                  </p>
                  <div className="space-y-4">
                    {[
                      "Gestion administrative et comptable rigoureuse",
                      "Suivi technique et maintenance des immeubles",
                      "Organisation et tenue des assemblées générales",
                      "Gestion des sinistres et contentieux",
                      "Accompagnement dans les projets de rénovation",
                    ].map((service, index) => (
                      <div key={index} className="flex items-start">
                        <div className="mt-1 mr-3 bg-[#002395] rounded-full p-1">
                          <Check className="h-4 w-4 text-white" />
                        </div>
                        <p>{service}</p>
                      </div>
                    ))}
                  </div>
                </div>
                <div className="relative h-[400px] rounded-2xl overflow-hidden">
                  <Image
                    src="https://images.unsplash.com/photo-1507003211169-0a1dd7228f2d?q=80&w=2070&auto=format&fit=crop"
                    alt="Syndic de copropriété à Paris"
                    fill
                    className="object-cover"
                  />
                </div>
              </div>
            </section>

            {/* Nos services de syndic de copropriété */}
            <section className="mb-12 py-12 bg-gray-50 -mx-8 px-8 rounded-xl" id="services">
              <h2 className="text-3xl font-bold mb-12 text-center text-[#002395]">
                Nos services de syndic de copropriété à Paris
              </h2>
              <div className="grid md:grid-cols-3 gap-8">
                {[
                  {
                    title: "Gestion Administrative",
                    description:
                      "Organisation des assemblées générales, tenue des registres, suivi des décisions, gestion des archives et des contrats.",
                    icon: <FileText className="h-10 w-10 text-white" />,
                  },
                  {
                    title: "Gestion Technique",
                    description:
                      "Suivi de l'entretien des parties communes, coordination des travaux, visites régulières de l'immeuble, gestion des urgences.",
                    icon: <Building2 className="h-10 w-10 text-white" />,
                  },
                  {
                    title: "Gestion Comptable",
                    description:
                      "Établissement du budget prévisionnel, appels de fonds, suivi des dépenses, recouvrement des impayés, tenue des comptes.",
                    icon: <Shield className="h-10 w-10 text-white" />,
                  },
                ].map((service, index) => (
                  <div key={index} className="bg-white rounded-xl shadow-lg p-8 border border-gray-100">
                    <div className="bg-gradient-to-r from-[#002395] to-[#001875] w-16 h-16 rounded-full flex items-center justify-center mb-6">
                      {service.icon}
                    </div>
                    <h3 className="text-xl font-bold mb-4">{service.title}</h3>
                    <p className="text-gray-700 mb-6">{service.description}</p>
                  </div>
                ))}
              </div>

              <div className="mt-12 grid md:grid-cols-2 gap-8">
                {[
                  {
                    title: "Gestion des Sinistres",
                    description:
                      "Déclaration et suivi des sinistres auprès des assurances, coordination des expertises et des travaux de remise en état.",
                    icon: <Shield className="h-10 w-10 text-white" />,
                  },
                  {
                    title: "Conseil et Accompagnement",
                    description:
                      "Conseil juridique et technique, accompagnement dans les projets de rénovation énergétique et de valorisation du patrimoine.",
                    icon: <Clock className="h-10 w-10 text-white" />,
                  },
                ].map((service, index) => (
                  <div key={index} className="bg-white rounded-xl shadow-lg p-8 border border-gray-100">
                    <div className="bg-gradient-to-r from-[#002395] to-[#001875] w-16 h-16 rounded-full flex items-center justify-center mb-6">
                      {service.icon}
                    </div>
                    <h3 className="text-xl font-bold mb-4">{service.title}</h3>
                    <p className="text-gray-700 mb-6">{service.description}</p>
                  </div>
                ))}
              </div>
            </section>

            {/* Pourquoi nous choisir */}
            <section className="mb-12" id="avantages">
              <h2 className="text-3xl font-bold mb-12 text-center text-[#002395]">
                Pourquoi choisir DOGES comme syndic de copropriété à Paris ?
              </h2>
              <div className="grid md:grid-cols-2 lg:grid-cols-4 gap-8">
                {[
                  {
                    title: "Proximité",
                    description:
                      "Une équipe locale dédiée qui connaît parfaitement les spécificités des immeubles parisiens.",
                  },
                  {
                    title: "Transparence",
                    description:
                      "Accès en ligne à tous les documents de la copropriété et communication régulière sur la gestion.",
                  },
                  {
                    title: "Réactivité",
                    description: "Intervention rapide en cas d'urgence et suivi personnalisé de chaque dossier.",
                  },
                  {
                    title: "Expertise",
                    description:
                      "Plus de 25 ans d'expérience dans la gestion de copropriétés à Paris et en Île-de-France.",
                  },
                ].map((advantage, index) => (
                  <div key={index} className="bg-white rounded-xl shadow-sm p-6 border border-gray-100">
                    <h3 className="text-xl font-semibold mb-3">{advantage.title}</h3>
                    <p className="text-gray-700">{advantage.description}</p>
                  </div>
                ))}
              </div>
            </section>

            {/* Tarifs */}
            <section className="mb-12" id="tarifs">
              <h2 className="text-3xl font-bold mb-6 text-[#002395]">Nos tarifs de syndic de copropriété à Paris</h2>
              <p className="text-lg text-gray-700 mb-6">
                Chez DOGES, nous proposons des tarifs transparents et adaptés à la taille et aux besoins spécifiques de
                votre copropriété. Nos honoraires sont calculés en fonction du nombre de lots, des équipements à gérer
                et des prestations choisies.
              </p>

              <div className="bg-white rounded-xl shadow-lg p-8 border border-gray-100 mb-8">
                <h3 className="text-xl font-bold mb-4">Forfait annuel tout inclus</h3>
                <p className="text-gray-700 mb-4">
                  Notre forfait comprend toutes les prestations courantes nécessaires à la bonne gestion de votre
                  copropriété :
                </p>
                <ul className="space-y-2 mb-6">
                  <li className="flex items-start">
                    <div className="mt-1 mr-3 bg-[#002395] rounded-full p-1">
                      <Check className="h-4 w-4 text-white" />
                    </div>
                    <p>Gestion administrative, comptable et technique</p>
                  </li>
                  <li className="flex items-start">
                    <div className="mt-1 mr-3 bg-[#002395] rounded-full p-1">
                      <Check className="h-4 w-4 text-white" />
                    </div>
                    <p>Organisation et tenue de l'assemblée générale annuelle</p>
                  </li>
                  <li className="flex items-start">
                    <div className="mt-1 mr-3 bg-[#002395] rounded-full p-1">
                      <Check className="h-4 w-4 text-white" />
                    </div>
                    <p>Visites régulières de l'immeuble</p>
                  </li>
                  <li className="flex items-start">
                    <div className="mt-1 mr-3 bg-[#002395] rounded-full p-1">
                      <Check className="h-4 w-4 text-white" />
                    </div>
                    <p>Accès à votre espace client en ligne</p>
                  </li>
                </ul>
                <p className="font-semibold">
                  À partir de 150€ HT par lot principal et par an (tarif indicatif, devis personnalisé sur demande)
                </p>
              </div>

              <div className="bg-white rounded-xl shadow-lg p-8 border border-gray-100">
                <h3 className="text-xl font-bold mb-4">Prestations particulières</h3>
                <p className="text-gray-700 mb-4">
                  Certaines prestations spécifiques font l'objet d'honoraires complémentaires, conformément au contrat
                  de syndic :
                </p>
                <ul className="space-y-2 mb-6">
                  <li className="flex items-start">
                    <div className="mt-1 mr-3 bg-[#002395] rounded-full p-1">
                      <Check className="h-4 w-4 text-white" />
                    </div>
                    <p>Organisation d'assemblées générales extraordinaires</p>
                  </li>
                  <li className="flex items-start">
                    <div className="mt-1 mr-3 bg-[#002395] rounded-full p-1">
                      <Check className="h-4 w-4 text-white" />
                    </div>
                    <p>Suivi de travaux exceptionnels</p>
                  </li>
                  <li className="flex items-start">
                    <div className="mt-1 mr-3 bg-[#002395] rounded-full p-1">
                      <Check className="h-4 w-4 text-white" />
                    </div>
                    <p>Gestion des contentieux</p>
                  </li>
                  <li className="flex items-start">
                    <div className="mt-1 mr-3 bg-[#002395] rounded-full p-1">
                      <Check className="h-4 w-4 text-white" />
                    </div>
                    <p>Prestations techniques spécifiques</p>
                  </li>
                </ul>
                <Link href="/contact">
                  <Button className="bg-[#002395] hover:bg-[#001875] text-white">
                    Demander un devis personnalisé
                    <ArrowRight className="ml-2 h-4 w-4" />
                  </Button>
                </Link>
              </div>
            </section>

            {/* FAQ pour le SEO */}
            <section className="mb-12 py-12 bg-gray-50 -mx-8 px-8 rounded-xl" id="faq">
              <h2 className="text-3xl font-bold mb-12 text-center text-[#002395]">
                Questions fréquentes sur le syndic de copropriété à Paris
              </h2>
              <div className="max-w-3xl mx-auto space-y-6">
                {[
                  {
                    question: "Comment changer de syndic de copropriété à Paris ?",
                    answer:
                      "Pour changer de syndic à Paris, il faut d'abord inscrire ce point à l'ordre du jour de la prochaine assemblée générale. Vous pouvez demander des devis à plusieurs syndics, dont DOGES, pour comparer les offres. Lors de l'AG, les copropriétaires votent pour élire le nouveau syndic à la majorité absolue (article 25 de la loi de 1965). Une fois élu, le nouveau syndic se charge des formalités de transfert avec l'ancien syndic.",
                  },
                  {
                    question: "Quelles sont les obligations d'un syndic de copropriété à Paris ?",
                    answer:
                      "Un syndic de copropriété à Paris doit respecter plusieurs obligations légales : représenter la copropriété dans tous les actes civils, exécuter les décisions d'assemblée générale, administrer l'immeuble, tenir la comptabilité du syndicat, établir le budget prévisionnel, assurer l'entretien et la conservation de l'immeuble, et veiller au respect du règlement de copropriété. À Paris, il doit également être particulièrement vigilant sur les questions liées aux réglementations locales et aux spécificités du bâti ancien.",
                  },
                  {
                    question: "Quel est le coût d'un syndic de copropriété à Paris ?",
                    answer:
                      "Le coût d'un syndic de copropriété à Paris varie selon plusieurs facteurs : taille de la copropriété, nombre de lots, équipements à gérer (ascenseur, chauffage collectif, etc.), et prestations incluses. En moyenne, pour une copropriété parisienne, les honoraires annuels se situent entre 150€ et 250€ HT par lot principal. Chez DOGES, nous proposons des tarifs transparents et compétitifs, adaptés aux spécificités de chaque immeuble parisien.",
                  },
                  {
                    question: "Quelles sont les spécificités des copropriétés parisiennes ?",
                    answer:
                      "Les copropriétés parisiennes présentent plusieurs spécificités : bâti ancien nécessitant un entretien régulier, contraintes architecturales et patrimoniales (immeubles classés ou en secteur sauvegardé), réglementations locales strictes (Plan Climat de Paris, encadrement des locations touristiques), forte densité d'occupation, et valorisation patrimoniale importante. Ces particularités requièrent l'expertise d'un syndic comme DOGES, qui connaît parfaitement le marché immobilier parisien et ses contraintes.",
                  },
                ].map((item, index) => (
                  <div key={index} className="bg-white rounded-lg shadow-sm p-6 border border-gray-100">
                    <h3 className="text-lg font-semibold mb-2">{item.question}</h3>
                    <p className="text-gray-700">{item.answer}</p>
                  </div>
                ))}
              </div>
            </section>

            {/* CTA */}
            <section className="py-12 bg-gradient-to-r from-[#002395] to-[#001875] text-white rounded-xl">
              <div className="px-6 text-center">
                <h2 className="text-3xl font-bold mb-6">Confiez la gestion de votre copropriété à DOGES</h2>
                <p className="text-xl mb-8 max-w-2xl mx-auto">
                  Bénéficiez d'une gestion professionnelle, transparente et efficace pour votre copropriété parisienne.
                </p>
                <div className="flex flex-col sm:flex-row justify-center gap-4">
                  <Link
                    href="/contact"
                    className="inline-flex items-center justify-center px-6 py-3 bg-white text-[#002395] font-medium rounded hover:bg-gray-100 transition-colors"
                  >
                    Prendre rendez-vous
                    <ArrowRight className="ml-2 h-4 w-4" />
                  </Link>
                  <Link
                    href="/contact"
                    className="inline-flex items-center justify-center px-6 py-3 bg-transparent text-white border-2 border-white font-medium rounded hover:bg-white hover:text-[#002395] transition-colors"
                  >
                    Demander un devis
                    <ArrowRight className="ml-2 h-4 w-4" />
                  </Link>
                </div>
              </div>
            </section>
          </div>

          {/* Barre latérale (1/3) */}
          <div className="lg:col-span-1">
            <div className="sticky top-24 space-y-8">
              {/* Table des matières */}
              <TableOfContentsWidget items={tocItems} />

              {/* Services associés */}
              <RelatedServicesWidget
                services={[
                  {
                    title: "Administrateur de Biens Paris",
                    description: "Gestion complète de votre patrimoine immobilier à Paris",
                    link: "/services/administrateur-biens-paris",
                  },
                  {
                    title: "Gestion Locative Paris",
                    description: "Gestion professionnelle de vos biens locatifs à Paris",
                    link: "/services/gestion-locative-paris",
                  },
                  {
                    title: "Garantie Loyers Impayés",
                    description: "Protégez vos revenus locatifs contre les impayés",
                    link: "/services/garantie-loyers-impayes",
                  },
                ]}
              />

              {/* Widget de contact */}
              <ContactWidget />
            </div>
          </div>
        </div>
      </div>

      {/* Schéma JSON-LD pour le SEO */}
      <script
        type="application/ld+json"
        dangerouslySetInnerHTML={{
          __html: JSON.stringify({
            "@context": "https://schema.org",
            "@type": "ProfessionalService",
            name: "DOGES - Syndic de Copropriété à Paris",
            url: "https://www.doges.fr/services/syndic-copropriete-paris",
            description:
              "DOGES, votre syndic de copropriété à Paris. Gestion transparente et efficace de votre copropriété parisienne avec un service personnalisé.",
            address: {
              "@type": "PostalAddress",
              streetAddress: "123 Avenue des Champs-Élysées",
              addressLocality: "Paris",
              postalCode: "75008",
              addressCountry: "FR",
            },
            geo: {
              "@type": "GeoCoordinates",
              latitude: "48.8698",
              longitude: "2.3075",
            },
            hasOfferCatalog: {
              "@type": "OfferCatalog",
              name: "Services de syndic de copropriété à Paris",
              itemListElement: [
                {
                  "@type": "Offer",
                  itemOffered: {
                    "@type": "Service",
                    name: "Gestion Administrative de Copropriété",
                    url: "https://www.doges.fr/services/syndic-copropriete-paris#services",
                  },
                },
                {
                  "@type": "Offer",
                  itemOffered: {
                    "@type": "Service",
                    name: "Gestion Technique d'Immeuble",
                    url: "https://www.doges.fr/services/syndic-copropriete-paris#services",
                  },
                },
                {
                  "@type": "Offer",
                  itemOffered: {
                    "@type": "Service",
                    name: "Gestion Comptable de Copropriété",
                    url: "https://www.doges.fr/services/syndic-copropriete-paris#services",
                  },
                },
              ],
            },
          }),
        }}
      />
    </>
  )
}
