"use client"

import { useState, useEffect } from "react"
import { Card, CardContent } from "@/components/ui/card"
import { Slider } from "@/components/ui/slider"

export default function TransactionCalculator() {
  const [activeTab, setActiveTab] = useState<"achat" | "vente">("achat")

  // États pour l'onglet Achat
  const [propertyPrice, setPropertyPrice] = useState(1430000)
  const [isNewProperty, setIsNewProperty] = useState(false)
  const [loanAmount, setLoanAmount] = useState(700000)
  const [loanPercentage, setLoanPercentage] = useState(700000 / 1430000)
  const [loanDuration, setLoanDuration] = useState(16)

  // États pour l'onglet Vente
  const [sellingPrice, setSellingPrice] = useState(1430000)
  const [purchasePrice, setPurchasePrice] = useState(1000000)
  const [purchaseYear, setPurchaseYear] = useState(2015)
  const [remainingLoan, setRemainingLoan] = useState(300000)

  const currentYear = new Date().getFullYear()
  const ownershipYears = currentYear - purchaseYear

  // Mettre à jour le montant du prêt lorsque le prix du bien change (onglet Achat)
  useEffect(() => {
    if (activeTab === "achat") {
      // Maintenir le même pourcentage de prêt par rapport au prix
      const newLoanAmount = Math.round(propertyPrice * loanPercentage)
      setLoanAmount(newLoanAmount)
    }
  }, [propertyPrice, loanPercentage, activeTab])

  // Gérer le changement du montant du prêt (onglet Achat)
  const handleLoanAmountChange = (value: number[]) => {
    const newLoanAmount = value[0]
    setLoanAmount(newLoanAmount)
    // Mettre à jour le pourcentage pour maintenir la cohérence
    setLoanPercentage(newLoanAmount / propertyPrice)
  }

  // Calculs pour l'achat
  const notaryFees = isNewProperty
    ? Math.round(propertyPrice * 0.03) // 3% pour le neuf
    : Math.round(propertyPrice * 0.08) // 8% pour l'ancien

  const agencyFees = Math.round(propertyPrice * 0.05) // 5% d'honoraires
  const totalCost = propertyPrice + notaryFees + agencyFees
  const personalContribution = totalCost - loanAmount

  // Estimation de la mensualité (très simplifiée)
  const monthlyPayment = Math.round((loanAmount * (1 + (0.03 * loanDuration) / 12)) / (loanDuration * 12))

  // Calculs pour la vente
  const sellingAgencyFees = Math.round(sellingPrice * 0.05) // 5% d'honoraires
  const capitalGain = sellingPrice - purchasePrice

  // Calcul simplifié de la taxe sur les plus-values
  let capitalGainsTax = 0
  if (capitalGain > 0 && ownershipYears < 30) {
    // Abattement pour durée de détention (simplifié)
    const reductionFactor = Math.min(1, ownershipYears / 30)
    const taxableGain = capitalGain * (1 - reductionFactor)
    capitalGainsTax = Math.round(taxableGain * 0.19) // Taux simplifié de 19%
  }

  const netSellerAmount = sellingPrice - sellingAgencyFees - remainingLoan - capitalGainsTax

  return (
    <Card className="shadow-md border-0" id="calculateur-detaille">
      <CardContent className="p-6">
        {/* Tabs */}
        <div className="grid grid-cols-2 gap-0 mb-8 h-14 rounded-md overflow-hidden">
          <button
            className={`h-full flex items-center justify-center font-medium text-lg ${
              activeTab === "achat" ? "bg-[#002395] text-white" : "bg-gray-100 text-gray-700 hover:bg-gray-200"
            }`}
            onClick={() => setActiveTab("achat")}
          >
            Achat
          </button>
          <button
            className={`h-full flex items-center justify-center font-medium text-lg ${
              activeTab === "vente" ? "bg-[#002395] text-white" : "bg-gray-100 text-gray-700 hover:bg-gray-200"
            }`}
            onClick={() => setActiveTab("vente")}
          >
            Vente
          </button>
        </div>

        {activeTab === "achat" ? (
          // Contenu de l'onglet Achat
          <div className="grid md:grid-cols-2 gap-8">
            {/* Partie gauche - Paramètres */}
            <div>
              {/* Prix du bien */}
              <div className="mb-8">
                <div className="flex justify-between mb-2">
                  <label className="font-medium text-lg">Prix du bien</label>
                  <span className="font-bold text-lg">{propertyPrice.toLocaleString()} €</span>
                </div>
                <Slider
                  defaultValue={[propertyPrice]}
                  min={50000}
                  max={2000000}
                  step={10000}
                  onValueChange={(value) => setPropertyPrice(value[0])}
                  className="my-4"
                />
                <div className="flex justify-between text-sm text-gray-500">
                  <span>50 000 €</span>
                  <span>2 000 000 €</span>
                </div>
              </div>

              {/* Type de bien */}
              <div className="mb-8">
                <label className="font-medium text-lg block mb-3">Type de bien</label>
                <div className="grid grid-cols-2 gap-4">
                  <button
                    className={`py-3 px-4 rounded font-medium ${
                      !isNewProperty
                        ? "bg-[#002395] text-white"
                        : "border border-gray-300 text-gray-700 hover:border-[#002395]"
                    }`}
                    onClick={() => setIsNewProperty(false)}
                  >
                    Ancien
                  </button>
                  <button
                    className={`py-3 px-4 rounded font-medium ${
                      isNewProperty
                        ? "bg-[#002395] text-white"
                        : "border border-gray-300 text-gray-700 hover:border-[#002395]"
                    }`}
                    onClick={() => setIsNewProperty(true)}
                  >
                    Neuf
                  </button>
                </div>
              </div>

              {/* Montant de l'emprunt */}
              <div className="mb-8">
                <div className="flex justify-between mb-2">
                  <label className="font-medium text-lg">Montant de l'emprunt</label>
                  <span className="font-bold text-lg">{loanAmount.toLocaleString()} €</span>
                </div>
                <Slider
                  value={[loanAmount]}
                  min={0}
                  max={propertyPrice}
                  step={10000}
                  onValueChange={handleLoanAmountChange}
                  className="my-4"
                />
                <div className="flex justify-between text-sm text-gray-500">
                  <span>0 €</span>
                  <span>{propertyPrice.toLocaleString()} €</span>
                </div>
              </div>

              {/* Durée du prêt */}
              <div className="mb-8">
                <div className="flex justify-between mb-2">
                  <label className="font-medium text-lg">Durée du prêt</label>
                  <span className="font-bold text-lg">{loanDuration} ans</span>
                </div>
                <Slider
                  defaultValue={[loanDuration]}
                  min={5}
                  max={30}
                  step={1}
                  onValueChange={(value) => setLoanDuration(value[0])}
                  className="my-4"
                />
                <div className="flex justify-between text-sm text-gray-500">
                  <span>5 ans</span>
                  <span>30 ans</span>
                </div>
              </div>
            </div>

            {/* Partie droite - Résultats */}
            <div className="bg-gray-50 p-6 rounded-lg">
              <h3 className="text-2xl font-bold mb-6 text-[#002395]">Résumé des coûts</h3>

              <div className="space-y-4 mb-8">
                <div className="flex justify-between">
                  <span className="text-lg">Prix d'achat</span>
                  <span className="text-lg font-bold">{propertyPrice.toLocaleString()} €</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-lg">Frais de notaire ({isNewProperty ? "3%" : "8%"})</span>
                  <span className="text-lg font-bold">{notaryFees.toLocaleString()} €</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-lg">Honoraires d'agence (5%)</span>
                  <span className="text-lg font-bold">{agencyFees.toLocaleString()} €</span>
                </div>
                <div className="flex justify-between pt-4 border-t border-gray-300">
                  <span className="text-xl font-bold">Coût total</span>
                  <span className="text-xl font-bold text-[#002395]">{totalCost.toLocaleString()} €</span>
                </div>
              </div>

              <h4 className="text-xl font-bold mb-4">Financement</h4>
              <div className="space-y-4 mb-6">
                <div className="flex justify-between">
                  <span className="text-lg">Apport personnel</span>
                  <span className="text-lg font-bold">{personalContribution.toLocaleString()} €</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-lg">Montant emprunté</span>
                  <span className="text-lg font-bold">{loanAmount.toLocaleString()} €</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-lg">Mensualité estimée*</span>
                  <span className="text-lg font-bold">{monthlyPayment.toLocaleString()} €</span>
                </div>
              </div>

              <p className="text-sm text-gray-500">
                *Estimation basée sur un taux d'intérêt moyen de 3%.
                <br />
                Contactez-nous pour une simulation personnalisée.
              </p>
            </div>
          </div>
        ) : (
          // Contenu de l'onglet Vente
          <div className="grid md:grid-cols-2 gap-8">
            {/* Partie gauche - Paramètres */}
            <div>
              {/* Prix de vente */}
              <div className="mb-8">
                <div className="flex justify-between mb-2">
                  <label className="font-medium text-lg">Prix de vente estimé</label>
                  <span className="font-bold text-lg">{sellingPrice.toLocaleString()} €</span>
                </div>
                <Slider
                  defaultValue={[sellingPrice]}
                  min={50000}
                  max={3000000}
                  step={10000}
                  onValueChange={(value) => setSellingPrice(value[0])}
                  className="my-4"
                />
                <div className="flex justify-between text-sm text-gray-500">
                  <span>50 000 €</span>
                  <span>3 000 000 €</span>
                </div>
              </div>

              {/* Prix d'achat initial */}
              <div className="mb-8">
                <div className="flex justify-between mb-2">
                  <label className="font-medium text-lg">Prix d'achat initial</label>
                  <span className="font-bold text-lg">{purchasePrice.toLocaleString()} €</span>
                </div>
                <Slider
                  defaultValue={[purchasePrice]}
                  min={10000}
                  max={sellingPrice}
                  step={10000}
                  onValueChange={(value) => setPurchasePrice(value[0])}
                  className="my-4"
                />
                <div className="flex justify-between text-sm text-gray-500">
                  <span>10 000 €</span>
                  <span>{sellingPrice.toLocaleString()} €</span>
                </div>
              </div>

              {/* Année d'achat */}
              <div className="mb-8">
                <div className="flex justify-between mb-2">
                  <label className="font-medium text-lg">Année d'achat</label>
                  <span className="font-bold text-lg">{purchaseYear}</span>
                </div>
                <Slider
                  defaultValue={[purchaseYear]}
                  min={1990}
                  max={currentYear - 1}
                  step={1}
                  onValueChange={(value) => setPurchaseYear(value[0])}
                  className="my-4"
                />
                <div className="flex justify-between text-sm text-gray-500">
                  <span>1990</span>
                  <span>{currentYear - 1}</span>
                </div>
              </div>

              {/* Reste à rembourser (crédit) */}
              <div className="mb-8">
                <div className="flex justify-between mb-2">
                  <label className="font-medium text-lg">Reste à rembourser (crédit)</label>
                  <span className="font-bold text-lg">{remainingLoan.toLocaleString()} €</span>
                </div>
                <Slider
                  defaultValue={[remainingLoan]}
                  min={0}
                  max={purchasePrice}
                  step={10000}
                  onValueChange={(value) => setRemainingLoan(value[0])}
                  className="my-4"
                />
                <div className="flex justify-between text-sm text-gray-500">
                  <span>0 €</span>
                  <span>{purchasePrice.toLocaleString()} €</span>
                </div>
              </div>
            </div>

            {/* Partie droite - Résultats */}
            <div className="bg-gray-50 p-6 rounded-lg">
              <h3 className="text-2xl font-bold mb-6 text-[#002395]">Résumé de la vente</h3>

              <div className="space-y-4 mb-8">
                <div className="flex justify-between">
                  <span className="text-lg">Prix de vente</span>
                  <span className="text-lg font-bold">{sellingPrice.toLocaleString()} €</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-lg">Prix d'achat initial</span>
                  <span className="text-lg font-bold">{purchasePrice.toLocaleString()} €</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-lg">Plus-value brute</span>
                  <span className="text-lg font-bold">{capitalGain.toLocaleString()} €</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-lg">Durée de détention</span>
                  <span className="text-lg font-bold">{ownershipYears} ans</span>
                </div>
              </div>

              <h4 className="text-xl font-bold mb-4">Frais et taxes</h4>
              <div className="space-y-4 mb-6">
                <div className="flex justify-between">
                  <span className="text-lg">Honoraires d'agence (5%)</span>
                  <span className="text-lg font-bold">{sellingAgencyFees.toLocaleString()} €</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-lg">Reste à rembourser (crédit)</span>
                  <span className="text-lg font-bold">{remainingLoan.toLocaleString()} €</span>
                </div>
                {capitalGainsTax > 0 && (
                  <div className="flex justify-between">
                    <span className="text-lg">Taxe sur plus-value*</span>
                    <span className="text-lg font-bold">{capitalGainsTax.toLocaleString()} €</span>
                  </div>
                )}
                <div className="flex justify-between pt-4 border-t border-gray-300">
                  <span className="text-xl font-bold">Net vendeur estimé</span>
                  <span className="text-xl font-bold text-[#002395]">{netSellerAmount.toLocaleString()} €</span>
                </div>
              </div>

              <p className="text-sm text-gray-500">
                *Estimation simplifiée. La fiscalité des plus-values immobilières dépend de nombreux facteurs.
                <br />
                Contactez-nous pour une analyse personnalisée de votre situation.
              </p>
            </div>
          </div>
        )}
      </CardContent>
    </Card>
  )
}
