"use client"

import type React from "react"

import Link from "next/link"
import { useState } from "react"
import { useRouter } from "next/navigation"
import { Input } from "@/components/ui/input"
import { Button } from "@/components/ui/button"
import SafeImage from "@/components/safe-image"

interface BlogSidebarWidgetsProps {
  relatedArticles?: {
    slug: string
    title: string
  }[]
}

// Export par défaut avec NOUVELLES images
export default function BlogSidebarWidgets({ relatedArticles }: BlogSidebarWidgetsProps) {
  const router = useRouter()
  const [searchQuery, setSearchQuery] = useState("")

  // Fonction de recherche optimisée
  const handleSearch = (e: React.FormEvent) => {
    e.preventDefault()
    if (searchQuery.trim()) {
      router.push(`/blog/search?q=${encodeURIComponent(searchQuery.trim())}`)
    }
  }

  // Articles populaires avec NOUVELLES images authentiques
  const popularArticles = [
    {
      slug: "guide-proprietaire-bailleur-2025",
      title: "Guide Complet du Propriétaire Bailleur 2025",
      date: "15 mars 2025",
      image: "/professional-property-management.jpg",
    },
    {
      slug: "investir-immobilier-ile-de-france-2025",
      title: "Où Investir dans l'Immobilier en Île-de-France en 2025 ?",
      date: "3 décembre 2024",
      image: "/paris-investment-2025.png", // NOUVELLE vue magnifique de Paris
    },
    {
      slug: "ifi-immobilier-locatif",
      title: "IFI et Immobilier Locatif : Stratégies d'Optimisation",
      date: "10 janvier 2025",
      image: "/ifi-tax-calculation.png", // NOUVELLE image de calcul fiscal
    },
  ]

  return (
    <div className="space-y-8">
      {/* Widget de recherche optimisé */}
      <div className="bg-white rounded-xl shadow-sm p-6 border border-gray-100">
        <h3 className="font-bold text-lg mb-4">Rechercher</h3>
        <form onSubmit={handleSearch} className="flex gap-2">
          <Input
            type="search"
            value={searchQuery}
            onChange={(e) => setSearchQuery(e.target.value)}
            placeholder="Rechercher un article..."
            className="flex-1"
            required
          />
          <Button type="submit">Rechercher</Button>
        </form>
      </div>

      {/* Widget des articles populaires avec NOUVELLES images */}
      <div className="bg-white rounded-xl shadow-sm p-6 border border-gray-100">
        <h3 className="font-bold text-lg mb-4">Articles populaires</h3>
        <div className="space-y-4">
          {popularArticles.map((article, i) => (
            <Link key={i} href={`/blog/articles/${article.slug}`} className="flex gap-3 group">
              <div className="w-16 h-16 relative flex-shrink-0">
                <SafeImage
                  src={article.image}
                  alt={article.title}
                  className="rounded-md object-cover"
                  width={64}
                  height={64}
                />
              </div>
              <div>
                <h4 className="text-sm font-medium group-hover:text-blue-600 line-clamp-2">{article.title}</h4>
                <p className="text-xs text-gray-500 mt-1">{article.date}</p>
              </div>
            </Link>
          ))}
        </div>
      </div>

      {/* Widget d'inscription à la newsletter */}
      <div className="bg-white rounded-xl shadow-sm p-6 border border-gray-100">
        <h3 className="font-bold text-lg mb-2">Newsletter</h3>
        <p className="text-sm text-gray-600 mb-4">
          Recevez nos conseils immobiliers et les dernières actualités du marché
        </p>
        <form className="space-y-3">
          <Input type="email" placeholder="Votre email" required />
          <Button type="submit" className="w-full">
            S'abonner
          </Button>
        </form>
      </div>
    </div>
  )
}
