"use client"

import { useState } from "react"
import { Button } from "@/components/ui/button"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Badge } from "@/components/ui/badge"

interface TestCase {
  name: string
  description: string
  steps: string[]
  expectedOutcome: string
}

const manualTestCases: TestCase[] = [
  {
    name: "Test Primo-Accédant Complet",
    description: "Tester le parcours complet d'un primo-accédant",
    steps: [
      "1. Cliquer sur le chatbot",
      "2. Choisir 'Acheter un bien'",
      "3. Sélectionner 'Oui, premier achat'",
      "4. Choisir un budget (ex: 500-800k€)",
      "5. Sélectionner une zone (ex: Paris intra-muros)",
      "6. Choisir le type (ex: 3 pièces)",
      "7. Indiquer le délai (ex: Dans les 6 mois)",
      "8. Vérifier les actions proposées (calculateur, RDV)",
    ],
    expectedOutcome: "Le chatbot doit qualifier le prospect et proposer des actions adaptées",
  },
  {
    name: "Test Locataire Rapide",
    description: "Tester la recherche de logement en location",
    steps: [
      "1. Ouvrir le chatbot",
      "2. Choisir 'Louer un logement'",
      "3. Indiquer le budget (ex: 1000€ à 1500€)",
      "4. Choisir la zone (ex: Paris est)",
      "5. Sélectionner le type (ex: 2 pièces)",
      "6. Préciser le délai (ex: Dans le mois)",
      "7. Vérifier la proposition d'alerte",
    ],
    expectedOutcome: "Proposition de création d'alerte personnalisée",
  },
  {
    name: "Test Vendeur Estimation",
    description: "Tester le parcours d'estimation pour un vendeur",
    steps: [
      "1. Lancer le chatbot",
      "2. Sélectionner 'Vendre un bien'",
      "3. Indiquer la localisation (ex: Paris intra-muros)",
      "4. Préciser le type (ex: Appartement)",
      "5. Donner la surface (ex: 50-80m²)",
      "6. Choisir le délai (ex: Délai normal)",
      "7. Vérifier les options d'estimation",
    ],
    expectedOutcome: "Proposition d'estimation gratuite et RDV expert",
  },
  {
    name: "Test Investisseur Avancé",
    description: "Tester le conseil en investissement",
    steps: [
      "1. Ouvrir le chatbot",
      "2. Choisir 'Investir'",
      "3. Sélectionner l'objectif (ex: Revenus locatifs)",
      "4. Indiquer le budget (ex: 400-600k€)",
      "5. Choisir la zone (ex: Paris)",
      "6. Sélectionner le type (ex: Appartement ancien)",
      "7. Vérifier les outils proposés",
    ],
    expectedOutcome: "Accès aux simulateurs de rentabilité et conseil personnalisé",
  },
  {
    name: "Test Message Libre",
    description: "Tester la compréhension de messages en langage naturel",
    steps: [
      "1. Ouvrir le chatbot",
      "2. Taper: 'Je cherche un studio à Paris pour 800€'",
      "3. Vérifier la détection automatique du profil",
      "4. Continuer la conversation",
      "5. Taper: 'Combien coûte la gestion locative ?'",
      "6. Vérifier la réponse de la base de connaissances",
    ],
    expectedOutcome: "Détection automatique + réponses pertinentes",
  },
  {
    name: "Test Professionnel",
    description: "Tester le parcours pour locaux professionnels",
    steps: [
      "1. Lancer le chatbot",
      "2. Choisir 'Autre' puis préciser 'bureaux'",
      "3. Ou taper directement 'Je cherche des bureaux'",
      "4. Suivre le flux professionnel",
      "5. Vérifier la proposition d'équipe dédiée",
    ],
    expectedOutcome: "Orientation vers l'équipe professionnelle",
  },
]

export default function ChatbotManualTest() {
  const [completedTests, setCompletedTests] = useState<Set<string>>(new Set())
  const [currentTest, setCurrentTest] = useState<string | null>(null)

  const markTestCompleted = (testName: string, success: boolean) => {
    if (success) {
      setCompletedTests((prev) => new Set([...prev, testName]))
    } else {
      setCompletedTests((prev) => {
        const newSet = new Set(prev)
        newSet.delete(testName)
        return newSet
      })
    }
    setCurrentTest(null)
  }

  const resetTests = () => {
    setCompletedTests(new Set())
    setCurrentTest(null)
  }

  return (
    <div className="p-6 space-y-6">
      <div className="flex justify-between items-center">
        <div>
          <h2 className="text-2xl font-bold">Tests Manuels du Chatbot</h2>
          <p className="text-gray-600">Testez chaque scénario en suivant les étapes</p>
        </div>
        <div className="flex items-center gap-4">
          <Badge variant="outline">
            {completedTests.size}/{manualTestCases.length} tests réussis
          </Badge>
          <Button onClick={resetTests} variant="outline">
            Réinitialiser
          </Button>
        </div>
      </div>

      {/* Barre de progression */}
      <div className="w-full bg-gray-200 rounded-full h-2">
        <div
          className="bg-green-600 h-2 rounded-full transition-all duration-300"
          style={{ width: `${(completedTests.size / manualTestCases.length) * 100}%` }}
        ></div>
      </div>

      {/* Tests manuels */}
      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        {manualTestCases.map((testCase) => {
          const isCompleted = completedTests.has(testCase.name)
          const isCurrent = currentTest === testCase.name

          return (
            <Card
              key={testCase.name}
              className={`border-2 transition-all ${
                isCompleted
                  ? "border-green-500 bg-green-50"
                  : isCurrent
                    ? "border-blue-500 bg-blue-50"
                    : "border-gray-200"
              }`}
            >
              <CardHeader>
                <CardTitle className="flex items-center justify-between">
                  <span className="flex items-center gap-2">
                    {isCompleted ? "✅" : isCurrent ? "🔄" : "⏳"}
                    {testCase.name}
                  </span>
                  <Badge variant={isCompleted ? "default" : "secondary"}>{isCompleted ? "Terminé" : "À tester"}</Badge>
                </CardTitle>
                <p className="text-sm text-gray-600">{testCase.description}</p>
              </CardHeader>
              <CardContent className="space-y-4">
                <div>
                  <h4 className="font-medium mb-2">Étapes à suivre:</h4>
                  <ol className="text-sm space-y-1">
                    {testCase.steps.map((step, index) => (
                      <li key={index} className="flex items-start gap-2">
                        <span className="text-gray-400 min-w-[20px]">{index + 1}.</span>
                        <span>{step.replace(/^\d+\.\s*/, "")}</span>
                      </li>
                    ))}
                  </ol>
                </div>

                <div className="bg-gray-50 p-3 rounded">
                  <h4 className="font-medium text-sm mb-1">Résultat attendu:</h4>
                  <p className="text-sm text-gray-700">{testCase.expectedOutcome}</p>
                </div>

                <div className="flex gap-2">
                  {!isCompleted && (
                    <Button
                      size="sm"
                      onClick={() => setCurrentTest(testCase.name)}
                      disabled={isCurrent}
                      className="flex-1"
                    >
                      {isCurrent ? "Test en cours..." : "Commencer le test"}
                    </Button>
                  )}

                  {isCurrent && (
                    <>
                      <Button
                        size="sm"
                        onClick={() => markTestCompleted(testCase.name, true)}
                        className="flex-1 bg-green-600 hover:bg-green-700"
                      >
                        ✅ Test réussi
                      </Button>
                      <Button
                        size="sm"
                        variant="outline"
                        onClick={() => markTestCompleted(testCase.name, false)}
                        className="flex-1"
                      >
                        ❌ Test échoué
                      </Button>
                    </>
                  )}

                  {isCompleted && (
                    <Button
                      size="sm"
                      variant="outline"
                      onClick={() => setCurrentTest(testCase.name)}
                      className="flex-1"
                    >
                      🔄 Retester
                    </Button>
                  )}
                </div>
              </CardContent>
            </Card>
          )
        })}
      </div>

      {/* Instructions générales */}
      <Card className="border-blue-500">
        <CardHeader>
          <CardTitle>Instructions de Test</CardTitle>
        </CardHeader>
        <CardContent className="space-y-3">
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div>
              <h4 className="font-medium mb-2">Points à vérifier :</h4>
              <ul className="text-sm space-y-1">
                <li>• Détection automatique du profil</li>
                <li>• Progression logique des étapes</li>
                <li>• Sauvegarde des données de qualification</li>
                <li>• Propositions d'actions pertinentes</li>
                <li>• Réponses de la base de connaissances</li>
              </ul>
            </div>
            <div>
              <h4 className="font-medium mb-2">En cas de problème :</h4>
              <ul className="text-sm space-y-1">
                <li>• Vérifier la console pour les erreurs</li>
                <li>• Tester avec différentes formulations</li>
                <li>• Vérifier que les étapes s'enchaînent</li>
                <li>• S'assurer que les actions fonctionnent</li>
                <li>• Valider l'affichage du profil détecté</li>
              </ul>
            </div>
          </div>
        </CardContent>
      </Card>
    </div>
  )
}
