"use client"

import type React from "react"

import { useState } from "react"
import { ArrowRight } from "lucide-react"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"

export default function DevisForm() {
  const [formData, setFormData] = useState({
    name: "",
    email: "",
    phone: "",
    propertyType: "",
    propertyAddress: "",
    propertySize: "",
    services: [] as string[],
    message: "",
    privacy: false,
  })

  const [isSubmitting, setIsSubmitting] = useState(false)
  const [submitResult, setSubmitResult] = useState<{ success: boolean; message: string } | null>(null)

  const handleChange = (e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement | HTMLSelectElement>) => {
    const { name, value } = e.target
    setFormData((prev) => ({ ...prev, [name]: value }))
  }

  const handleCheckboxChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const { name, checked } = e.target
    setFormData((prev) => ({ ...prev, [name]: checked }))
  }

  const handleServiceChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const { value, checked } = e.target
    setFormData((prev) => ({
      ...prev,
      services: checked ? [...prev.services, value] : prev.services.filter((service) => service !== value),
    }))
  }

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault()
    setIsSubmitting(true)
    setSubmitResult(null)

    try {
      // Simuler un appel API
      await new Promise((resolve) => setTimeout(resolve, 1000))

      // Simuler une réponse réussie
      setSubmitResult({
        success: true,
        message: "Votre demande de devis a été envoyée avec succès. Nous vous contacterons dans les plus brefs délais.",
      })

      // Réinitialiser le formulaire
      setFormData({
        name: "",
        email: "",
        phone: "",
        propertyType: "",
        propertyAddress: "",
        propertySize: "",
        services: [],
        message: "",
        privacy: false,
      })
    } catch (error) {
      setSubmitResult({
        success: false,
        message: "Une erreur est survenue lors de l'envoi de votre demande. Veuillez réessayer.",
      })
    } finally {
      setIsSubmitting(false)
    }
  }

  return (
    <form className="space-y-6" onSubmit={handleSubmit}>
      {submitResult && (
        <div
          className={`mb-6 p-4 rounded-lg ${
            submitResult.success ? "bg-green-50 text-green-700" : "bg-red-50 text-red-700"
          }`}
        >
          {submitResult.message}
        </div>
      )}

      <div className="grid grid-cols-2 gap-6">
        <div>
          <Label htmlFor="name">Nom complet</Label>
          <Input
            id="name"
            name="name"
            type="text"
            placeholder="Votre nom"
            required
            value={formData.name}
            onChange={handleChange}
          />
        </div>
        <div>
          <Label htmlFor="email">Email</Label>
          <Input
            id="email"
            name="email"
            type="email"
            placeholder="votre@email.com"
            required
            value={formData.email}
            onChange={handleChange}
          />
        </div>
      </div>

      <div>
        <Label htmlFor="phone">Téléphone</Label>
        <Input
          id="phone"
          name="phone"
          type="tel"
          placeholder="Votre numéro de téléphone"
          required
          value={formData.phone}
          onChange={handleChange}
        />
      </div>

      <div>
        <Label htmlFor="propertyType">Type de bien</Label>
        <select
          id="propertyType"
          name="propertyType"
          className="w-full h-9 rounded-md border border-input bg-transparent px-3 py-1 text-sm shadow-sm transition-colors focus-visible:outline-none focus-visible:ring-1 focus-visible:ring-ring"
          required
          value={formData.propertyType}
          onChange={handleChange}
        >
          <option value="">Sélectionnez un type de bien</option>
          <option value="appartement">Appartement</option>
          <option value="maison">Maison</option>
          <option value="commerce">Local commercial</option>
          <option value="bureau">Bureau</option>
          <option value="immeuble">Immeuble</option>
          <option value="autre">Autre</option>
        </select>
      </div>

      <div>
        <Label htmlFor="propertyAddress">Adresse du bien</Label>
        <Input
          id="propertyAddress"
          name="propertyAddress"
          type="text"
          placeholder="Adresse complète du bien"
          value={formData.propertyAddress}
          onChange={handleChange}
        />
      </div>

      <div>
        <Label htmlFor="propertySize">Surface (m²)</Label>
        <Input
          id="propertySize"
          name="propertySize"
          type="number"
          placeholder="Surface en m²"
          value={formData.propertySize}
          onChange={handleChange}
        />
      </div>

      <div>
        <Label className="block mb-2">Services souhaités</Label>
        <div className="grid grid-cols-2 gap-2">
          <div className="flex items-center">
            <input
              type="checkbox"
              id="gestion-complete"
              name="services"
              value="gestion-complete"
              className="mr-2"
              checked={formData.services.includes("gestion-complete")}
              onChange={handleServiceChange}
            />
            <Label htmlFor="gestion-complete" className="text-sm">
              Gestion complète
            </Label>
          </div>
          <div className="flex items-center">
            <input
              type="checkbox"
              id="recherche-locataire"
              name="services"
              value="recherche-locataire"
              className="mr-2"
              checked={formData.services.includes("recherche-locataire")}
              onChange={handleServiceChange}
            />
            <Label htmlFor="recherche-locataire" className="text-sm">
              Recherche de locataire
            </Label>
          </div>
          <div className="flex items-center">
            <input
              type="checkbox"
              id="gli"
              name="services"
              value="gli"
              className="mr-2"
              checked={formData.services.includes("gli")}
              onChange={handleServiceChange}
            />
            <Label htmlFor="gli" className="text-sm">
              Garantie Loyers Impayés
            </Label>
          </div>
          <div className="flex items-center">
            <input
              type="checkbox"
              id="pno"
              name="services"
              value="pno"
              className="mr-2"
              checked={formData.services.includes("pno")}
              onChange={handleServiceChange}
            />
            <Label htmlFor="pno" className="text-sm">
              Assurance PNO
            </Label>
          </div>
        </div>
      </div>

      <div>
        <Label htmlFor="message">Précisions supplémentaires</Label>
        <textarea
          id="message"
          name="message"
          className="w-full min-h-[100px] rounded-md border border-input bg-transparent px-3 py-2 text-sm shadow-sm transition-colors focus-visible:outline-none focus-visible:ring-1 focus-visible:ring-ring"
          placeholder="Informations complémentaires pour votre demande de devis..."
          value={formData.message}
          onChange={handleChange}
        ></textarea>
      </div>

      <div className="flex items-center">
        <input
          type="checkbox"
          id="privacy"
          name="privacy"
          className="mr-2"
          required
          checked={formData.privacy}
          onChange={handleCheckboxChange}
        />
        <Label htmlFor="privacy" className="text-sm text-gray-600">
          J'accepte que mes données soient traitées conformément à la{" "}
          <a href="/politique-confidentialite" className="text-[#002395] hover:underline">
            politique de confidentialité
          </a>
        </Label>
      </div>

      <Button type="submit" className="bg-[#002395] hover:bg-[#001875] text-white w-full" disabled={isSubmitting}>
        {isSubmitting ? "Envoi en cours..." : "Demander un devis gratuit"}
        <ArrowRight className="ml-2 h-4 w-4" />
      </Button>
    </form>
  )
}
