"use client"

import { useState } from "react"
import { Button } from "@/components/ui/button"
import { searchKnowledge, getKnowledgeStats } from "@/lib/chatbot-knowledge-base"
import { MessageCircle, CheckCircle, XCircle } from "lucide-react"

interface TestQuestion {
  id: string
  question: string
  expectedCategory: string
  keywords: string[]
}

const faqTestQuestions: TestQuestion[] = [
  {
    id: "test1",
    question: "Quels sont les services inclus dans la gestion locative ?",
    expectedCategory: "gestion",
    keywords: ["gestion locative", "services", "inclus"],
  },
  {
    id: "test2",
    question: "Comment est calculée votre commission sur les ventes ?",
    expectedCategory: "transaction",
    keywords: ["commission", "vente", "pourcentage"],
  },
  {
    id: "test3",
    question: "Comment fonctionne la garantie loyers impayés ?",
    expectedCategory: "gli",
    keywords: ["garantie", "loyers impayés", "GLI"],
  },
  {
    id: "test4",
    question: "Qu'est-ce que l'assurance PNO ?",
    expectedCategory: "pno",
    keywords: ["assurance PNO", "propriétaire non occupant"],
  },
  {
    id: "test5",
    question: "Quelle est la durée d'engagement pour la gestion ?",
    expectedCategory: "gestion",
    keywords: ["durée", "engagement", "mandat"],
  },
  {
    id: "test6",
    question: "Comment gérez-vous l'encadrement des loyers à Paris ?",
    expectedCategory: "gestion",
    keywords: ["encadrement loyers", "Paris", "réglementation"],
  },
  {
    id: "test7",
    question: "Quelles sont les spécificités du 16ème arrondissement ?",
    expectedCategory: "gestion",
    keywords: ["16ème arrondissement", "spécificités"],
  },
  {
    id: "test8",
    question: "Comment accompagnez-vous pour la mise en conformité DPE ?",
    expectedCategory: "gestion",
    keywords: ["DPE", "conformité", "passoires thermiques"],
  },
  {
    id: "test9",
    question: "Quelles garanties pour les logements de luxe ?",
    expectedCategory: "gli",
    keywords: ["logements luxe", "garanties", "haut de gamme"],
  },
  {
    id: "test10",
    question: "Comment gérez-vous les locations Airbnb ?",
    expectedCategory: "gestion",
    keywords: ["Airbnb", "locations saisonnières", "conformité"],
  },
]

interface TestResult {
  question: string
  found: boolean
  response: string
  source: string
  category: string
  relevanceScore: number
}

export default function FAQChatbotTester() {
  const [testResults, setTestResults] = useState<TestResult[]>([])
  const [isRunning, setIsRunning] = useState(false)
  const [currentTest, setCurrentTest] = useState<number>(0)
  const [manualQuestion, setManualQuestion] = useState("")
  const [manualResult, setManualResult] = useState<any>(null)

  const runAllTests = async () => {
    setIsRunning(true)
    setTestResults([])
    setCurrentTest(0)

    for (let i = 0; i < faqTestQuestions.length; i++) {
      setCurrentTest(i + 1)
      const testQuestion = faqTestQuestions[i]

      // Simuler un délai pour voir la progression
      await new Promise((resolve) => setTimeout(resolve, 500))

      const results = searchKnowledge(testQuestion.question)

      const testResult: TestResult = {
        question: testQuestion.question,
        found: results.length > 0,
        response: results.length > 0 ? results[0].content : "Aucune réponse trouvée",
        source: results.length > 0 ? results[0].source || "unknown" : "none",
        category: results.length > 0 ? results[0].category : "none",
        relevanceScore: calculateRelevanceScore(testQuestion, results[0]),
      }

      setTestResults((prev) => [...prev, testResult])
    }

    setIsRunning(false)
    setCurrentTest(0)
  }

  const calculateRelevanceScore = (testQuestion: TestQuestion, result: any): number => {
    if (!result) return 0

    let score = 0
    const resultText = result.content.toLowerCase()

    // Vérifier la présence des mots-clés attendus
    testQuestion.keywords.forEach((keyword) => {
      if (resultText.includes(keyword.toLowerCase())) {
        score += 20
      }
    })

    // Bonus si la catégorie correspond
    if (result.category === testQuestion.expectedCategory) {
      score += 20
    }

    // Bonus si c'est une réponse FAQ (plus détaillée)
    if (result.source === "faq") {
      score += 20
    }

    return Math.min(score, 100)
  }

  const testManualQuestion = () => {
    if (!manualQuestion.trim()) return

    const results = searchKnowledge(manualQuestion)
    setManualResult({
      question: manualQuestion,
      results: results,
      timestamp: new Date(),
    })
  }

  const stats = getKnowledgeStats()
  const successfulTests = testResults.filter((r) => r.found && r.relevanceScore >= 60).length
  const totalTests = testResults.length
  const successRate = totalTests > 0 ? ((successfulTests / totalTests) * 100).toFixed(1) : 0

  return (
    <div className="max-w-6xl mx-auto p-6 space-y-6">
      <div className="bg-white rounded-lg shadow-lg p-6">
        <h1 className="text-3xl font-bold text-[#002395] mb-6 flex items-center gap-3">
          <MessageCircle className="h-8 w-8" />
          Test du Chatbot avec Questions FAQ
        </h1>

        {/* Statistiques de la base de connaissances */}
        <div className="grid grid-cols-1 md:grid-cols-4 gap-4 mb-6">
          <div className="bg-blue-50 p-4 rounded-lg">
            <div className="text-2xl font-bold text-[#002395]">{stats.total}</div>
            <div className="text-sm text-gray-600">Total éléments</div>
          </div>
          <div className="bg-green-50 p-4 rounded-lg">
            <div className="text-2xl font-bold text-green-600">{stats.faqItems}</div>
            <div className="text-sm text-gray-600">Questions FAQ</div>
          </div>
          <div className="bg-purple-50 p-4 rounded-lg">
            <div className="text-2xl font-bold text-purple-600">{stats.knowledgeItems}</div>
            <div className="text-sm text-gray-600">Base générale</div>
          </div>
          <div className="bg-orange-50 p-4 rounded-lg">
            <div className="text-2xl font-bold text-orange-600">{stats.categories.length}</div>
            <div className="text-sm text-gray-600">Catégories</div>
          </div>
        </div>

        {/* Tests automatisés */}
        <div className="mb-8">
          <div className="flex items-center justify-between mb-4">
            <h2 className="text-xl font-semibold">Tests Automatisés FAQ</h2>
            <Button onClick={runAllTests} disabled={isRunning} className="bg-[#002395] hover:bg-[#001875]">
              {isRunning ? `Test ${currentTest}/${faqTestQuestions.length}...` : "🚀 Lancer tous les tests"}
            </Button>
          </div>

          {isRunning && (
            <div className="mb-4">
              <div className="bg-gray-200 rounded-full h-2">
                <div
                  className="bg-[#002395] h-2 rounded-full transition-all duration-300"
                  style={{ width: `${(currentTest / faqTestQuestions.length) * 100}%` }}
                ></div>
              </div>
              <p className="text-sm text-gray-600 mt-2">
                Test en cours : {currentTest}/{faqTestQuestions.length}
              </p>
            </div>
          )}

          {testResults.length > 0 && (
            <div className="mb-4 p-4 bg-gray-50 rounded-lg">
              <h3 className="font-semibold mb-2">Résultats Globaux</h3>
              <div className="grid grid-cols-3 gap-4 text-center">
                <div>
                  <div className="text-2xl font-bold text-green-600">{successfulTests}</div>
                  <div className="text-sm">Tests réussis</div>
                </div>
                <div>
                  <div className="text-2xl font-bold text-red-600">{totalTests - successfulTests}</div>
                  <div className="text-sm">Tests échoués</div>
                </div>
                <div>
                  <div className="text-2xl font-bold text-[#002395]">{successRate}%</div>
                  <div className="text-sm">Taux de réussite</div>
                </div>
              </div>
            </div>
          )}

          <div className="space-y-3">
            {testResults.map((result, index) => (
              <div key={index} className="border rounded-lg p-4">
                <div className="flex items-start justify-between mb-2">
                  <div className="flex items-center gap-2">
                    {result.found && result.relevanceScore >= 60 ? (
                      <CheckCircle className="h-5 w-5 text-green-500" />
                    ) : (
                      <XCircle className="h-5 w-5 text-red-500" />
                    )}
                    <span className="font-medium">Question {index + 1}</span>
                    <span className="text-xs bg-gray-100 px-2 py-1 rounded">Score: {result.relevanceScore}%</span>
                    <span className="text-xs bg-blue-100 px-2 py-1 rounded">{result.source}</span>
                  </div>
                </div>

                <div className="mb-2">
                  <strong>Q:</strong> {result.question}
                </div>

                <div className="bg-gray-50 p-3 rounded text-sm">
                  <strong>R:</strong> {result.response.substring(0, 200)}
                  {result.response.length > 200 && "..."}
                </div>
              </div>
            ))}
          </div>
        </div>

        {/* Test manuel */}
        <div className="border-t pt-6">
          <h2 className="text-xl font-semibold mb-4">Test Manuel</h2>
          <div className="flex gap-2 mb-4">
            <input
              type="text"
              value={manualQuestion}
              onChange={(e) => setManualQuestion(e.target.value)}
              placeholder="Posez votre question ici..."
              className="flex-1 p-3 border rounded-lg"
              onKeyPress={(e) => e.key === "Enter" && testManualQuestion()}
            />
            <Button onClick={testManualQuestion} className="bg-[#002395] hover:bg-[#001875]">
              Tester
            </Button>
          </div>

          {manualResult && (
            <div className="bg-gray-50 p-4 rounded-lg">
              <div className="mb-3">
                <strong>Question:</strong> {manualResult.question}
              </div>

              {manualResult.results.length > 0 ? (
                <div className="space-y-3">
                  {manualResult.results.map((result: any, index: number) => (
                    <div key={index} className="bg-white p-3 rounded border">
                      <div className="flex items-center gap-2 mb-2">
                        <span className="text-xs bg-blue-100 px-2 py-1 rounded">{result.source || "knowledge"}</span>
                        <span className="text-xs bg-gray-100 px-2 py-1 rounded">{result.category}</span>
                      </div>
                      <div className="text-sm">{result.content}</div>
                    </div>
                  ))}
                </div>
              ) : (
                <div className="text-red-600 flex items-center gap-2">
                  <XCircle className="h-4 w-4" />
                  Aucune réponse trouvée dans la base de connaissances
                </div>
              )}
            </div>
          )}
        </div>

        {/* Questions suggérées */}
        <div className="border-t pt-6">
          <h2 className="text-xl font-semibold mb-4">Questions Suggérées à Tester</h2>
          <div className="grid grid-cols-1 md:grid-cols-2 gap-3">
            {faqTestQuestions.slice(0, 6).map((q, index) => (
              <button
                key={index}
                onClick={() => setManualQuestion(q.question)}
                className="text-left p-3 border rounded-lg hover:bg-gray-50 transition-colors"
              >
                <div className="font-medium text-sm">{q.question}</div>
                <div className="text-xs text-gray-500 mt-1">Mots-clés: {q.keywords.join(", ")}</div>
              </button>
            ))}
          </div>
        </div>
      </div>
    </div>
  )
}
