"use client"

import { useEffect } from "react"

export interface HowToStep {
  name: string
  text: string
  url?: string
  imageUrl?: string
}

export interface HowToTool {
  name: string
  description?: string
}

export interface HowToSupply {
  name: string
  description?: string
}

interface HowToSchemaProps {
  name: string
  description: string
  totalTime?: string // Format ISO 8601 duration, ex: "PT2H30M" (2 heures 30 minutes)
  estimatedCost?: {
    currency: string
    value: string
  }
  steps: HowToStep[]
  tools?: HowToTool[]
  supplies?: HowToSupply[]
  imageUrl?: string
}

export function HowToSchema({
  name,
  description,
  totalTime,
  estimatedCost,
  steps,
  tools,
  supplies,
  imageUrl,
}: HowToSchemaProps) {
  useEffect(() => {
    if (!steps || steps.length === 0) return

    const howToSchema: any = {
      "@context": "https://schema.org",
      "@type": "HowTo",
      name: name,
      description: description,
      step: steps.map((step, index) => ({
        "@type": "HowToStep",
        position: index + 1,
        name: step.name,
        text: step.text,
        ...(step.url && { url: step.url }),
        ...(step.imageUrl && {
          image: {
            "@type": "ImageObject",
            url: step.imageUrl,
          },
        }),
      })),
    }

    // Ajouter les propriétés optionnelles si elles sont fournies
    if (totalTime) {
      howToSchema.totalTime = totalTime
    }

    if (estimatedCost) {
      howToSchema.estimatedCost = {
        "@type": "MonetaryAmount",
        currency: estimatedCost.currency,
        value: estimatedCost.value,
      }
    }

    if (tools && tools.length > 0) {
      howToSchema.tool = tools.map((tool) => ({
        "@type": "HowToTool",
        name: tool.name,
        ...(tool.description && { description: tool.description }),
      }))
    }

    if (supplies && supplies.length > 0) {
      howToSchema.supply = supplies.map((supply) => ({
        "@type": "HowToSupply",
        name: supply.name,
        ...(supply.description && { description: supply.description }),
      }))
    }

    if (imageUrl) {
      howToSchema.image = {
        "@type": "ImageObject",
        url: imageUrl,
      }
    }

    // Créer et ajouter le script JSON-LD
    const script = document.createElement("script")
    script.type = "application/ld+json"
    script.text = JSON.stringify(howToSchema)
    script.id = "howto-schema"

    // Supprimer tout script existant avec le même ID
    const existingScript = document.getElementById("howto-schema")
    if (existingScript) {
      document.head.removeChild(existingScript)
    }

    document.head.appendChild(script)

    // Nettoyer lors du démontage du composant
    return () => {
      const scriptToRemove = document.getElementById("howto-schema")
      if (scriptToRemove) {
        document.head.removeChild(scriptToRemove)
      }
    }
  }, [name, description, totalTime, estimatedCost, steps, tools, supplies, imageUrl])

  return null
}
