"use client"

import type React from "react"
import { useState, useEffect, useRef } from "react"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import {
  MessageCircle,
  X,
  Send,
  Calculator,
  Calendar,
  MapPin,
  TrendingUp,
  Home,
  Building,
  Phone,
  Mail,
  Shield,
  FileText,
  Scale,
} from "lucide-react"
import { searchKnowledge } from "@/lib/chatbot-knowledge-base"
import { ProfileDetector, type UserProfile, type UserContext } from "@/lib/chatbot-profile-detector"
import { conversationFlows, getNextStep, getWelcomeMessage } from "@/lib/chatbot-conversation-flows"

interface Message {
  id: string
  text: string
  sender: "user" | "bot"
  timestamp: Date
  options?: string[]
  actions?: ChatAction[]
  isContactInfo?: boolean
}

interface ChatAction {
  type: "calculator" | "appointment" | "estimation" | "alert" | "simulation" | "contact" | "service"
  label: string
  icon: React.ReactNode
  onClick: () => void
}

interface ConversationState {
  currentStep: string
  profile: UserProfile
  qualificationData: Record<string, any>
  conversationHistory: string[]
  showContactInfo?: boolean
  serviceInterest?: string
}

interface ContactInfo {
  name: string
  role: string
  phone: string
  email: string
}

// Informations de contact de l'agence
const agencyContacts: Record<string, ContactInfo> = {
  acheteur_primo: {
    name: "Bedis",
    role: "Conseiller Transaction",
    phone: "01 84 80 98 10",
    email: "contact@dogesadb.fr",
  },
  vendeur: {
    name: "Bedis",
    role: "Expert Estimation",
    phone: "01 84 80 98 10",
    email: "contact@dogesadb.fr",
  },
  locataire: {
    name: "Bedis",
    role: "Conseiller Location",
    phone: "01 84 80 98 10",
    email: "contact@dogesadb.fr",
  },
  proprietaire_bailleur: {
    name: "Bedis",
    role: "Spécialiste Gestion Locative",
    phone: "01 84 80 98 10",
    email: "contact@dogesadb.fr",
  },
  investisseur: {
    name: "Bedis",
    role: "Conseiller Investissement",
    phone: "01 84 80 98 10",
    email: "contact@dogesadb.fr",
  },
  professionnel: {
    name: "Bedis",
    role: "Responsable Professionnels",
    phone: "01 84 80 98 10",
    email: "contact@dogesadb.fr",
  },
  service_inquiry: {
    name: "Bedis",
    role: "Conseiller Services",
    phone: "01 84 80 98 10",
    email: "contact@dogesadb.fr",
  },
  "administrateur-biens-paris": {
    name: "Bedis",
    role: "Administrateur de Biens Paris",
    phone: "01 84 80 98 10",
    email: "admin-biens@dogesadb.fr",
  },
  "garantie-loyers-impayes": {
    name: "Bedis",
    role: "Spécialiste GLI",
    phone: "01 84 80 98 10",
    email: "gli@dogesadb.fr",
  },
  "assurance-pno": {
    name: "Bedis",
    role: "Conseiller Assurances",
    phone: "01 84 80 98 10",
    email: "assurances@dogesadb.fr",
  },
  unknown: {
    name: "Bedis",
    role: "Service Client",
    phone: "01 84 80 98 10",
    email: "contact@dogesadb.fr",
  },
}

export default function IntelligentChatbot() {
  const [isOpen, setIsOpen] = useState(false)
  const [messages, setMessages] = useState<Message[]>([])
  const [input, setInput] = useState("")
  const [isTyping, setIsTyping] = useState(false)
  const [hasNewMessage, setHasNewMessage] = useState(false)

  // État de conversation structuré
  const [conversationState, setConversationState] = useState<ConversationState>({
    currentStep: "welcome",
    profile: "unknown",
    qualificationData: {},
    conversationHistory: [],
    showContactInfo: false,
  })

  const profileDetector = useRef(new ProfileDetector())
  const [userContext, setUserContext] = useState<UserContext>(profileDetector.current.getContext())
  const messagesEndRef = useRef<HTMLDivElement>(null)

  const scrollToBottom = () => {
    messagesEndRef.current?.scrollIntoView({ behavior: "smooth" })
  }

  useEffect(() => {
    scrollToBottom()
  }, [messages])

  useEffect(() => {
    if (isOpen && messages.length === 0) {
      // Message d'accueil initial
      addBotMessage(
        "Bonjour ! Je suis l'assistant intelligent de DOGES. Je vais vous aider à trouver exactement ce que vous cherchez. Quel est votre projet immobilier ? 🏠",
        ["Acheter un bien", "Vendre un bien", "Louer un logement", "Investir", "Gestion locative", "Nos services"],
      )
    }
  }, [isOpen])

  // Effet pour l'animation subtile du bouton
  useEffect(() => {
    if (!isOpen) {
      // Activer l'indicateur de nouveau message après 30 secondes
      const timer = setTimeout(() => {
        setHasNewMessage(true)
      }, 30000)
      return () => clearTimeout(timer)
    } else {
      setHasNewMessage(false)
    }
  }, [isOpen])

  const addBotMessage = (text: string, options?: string[], actions?: ChatAction[], isContactInfo?: boolean) => {
    setIsTyping(true)
    setTimeout(() => {
      const newMessage: Message = {
        id: Date.now().toString(),
        text,
        sender: "bot",
        timestamp: new Date(),
        options,
        actions,
        isContactInfo,
      }
      setMessages((prev) => [...prev, newMessage])
      setIsTyping(false)
    }, 1000)
  }

  const addUserMessage = (text: string) => {
    const newMessage: Message = {
      id: Date.now().toString(),
      text,
      sender: "user",
      timestamp: new Date(),
    }
    setMessages((prev) => [...prev, newMessage])

    // Ajouter à l'historique de conversation
    setConversationState((prev) => ({
      ...prev,
      conversationHistory: [...prev.conversationHistory, text],
    }))
  }

  const updateConversationState = (updates: Partial<ConversationState>) => {
    setConversationState((prev) => ({ ...prev, ...updates }))
  }

  const handleProfileDetection = (message: string): UserProfile => {
    const detectedProfile = profileDetector.current.analyzeMessage(message)
    const context = profileDetector.current.getContext()
    setUserContext(context)

    // Extraire des données de qualification
    const budget = profileDetector.current.extractBudget(message)
    const zone = profileDetector.current.extractZone(message)

    const newQualificationData = { ...conversationState.qualificationData }
    if (budget) newQualificationData.budget = budget
    if (zone) newQualificationData.zone = zone

    // Mettre à jour l'état de conversation
    updateConversationState({
      profile: detectedProfile,
      qualificationData: newQualificationData,
      serviceInterest: context.serviceInterest,
    })

    return detectedProfile
  }

  const processConversationFlow = (userInput: string, selectedOption?: string): void => {
    const currentProfile = conversationState.profile
    const currentStepId = conversationState.currentStep

    // Si on a une option sélectionnée, traiter selon le flux
    if (selectedOption) {
      handleFlowProgression(selectedOption)
      return
    }

    // Sinon, analyser le message libre
    const detectedProfile = handleProfileDetection(userInput)
    const serviceInterest = userContext.serviceInterest

    // Si le profil a changé, rediriger vers le bon flux
    if (detectedProfile !== currentProfile && detectedProfile !== "unknown") {
      const welcomeStep = getWelcomeMessage(detectedProfile)
      updateConversationState({
        profile: detectedProfile,
        currentStep: "welcome",
      })

      setTimeout(() => {
        addBotMessage(welcomeStep.message, welcomeStep.options, getActionsForProfile(detectedProfile, serviceInterest))
      }, 1500)
      return
    }

    // Si c'est une demande de service spécifique
    if (detectedProfile === "service_inquiry" && serviceInterest) {
      handleServiceInquiry(serviceInterest)
      return
    }

    // Rechercher dans la base de connaissances
    const knowledgeResults = searchKnowledge(userInput)
    if (knowledgeResults.length > 0) {
      const response = knowledgeResults[0].content
      setTimeout(() => {
        addBotMessage(`${response}\n\nSouhaitez-vous que nous continuions avec votre projet ?`, [
          "Oui, continuons",
          "J'ai d'autres questions",
          "Prendre rendez-vous",
        ])
      }, 1500)
      return
    }

    // Réponse par défaut
    setTimeout(() => {
      addBotMessage("Je comprends. Pouvez-vous me donner plus de détails sur ce que vous recherchez ?", [
        "Préciser mon projet",
        "Parler à un conseiller",
        "Voir nos services",
      ])
    }, 1500)
  }

  const handleServiceInquiry = (serviceType: string) => {
    // Réponses spécifiques selon le type de service demandé
    if (serviceType.includes("administrateur-biens")) {
      let message =
        "Notre service d'Administration de Biens assure la gestion complète de votre patrimoine immobilier : gestion locative, comptabilité, suivi technique, valorisation patrimoniale."

      if (serviceType.includes("paris")) {
        if (serviceType.includes("13-14-15")) {
          message +=
            "\n\nNotre expertise dans les 13e, 14e et 15e arrondissements de Paris comprend une connaissance approfondie de ces quartiers dynamiques et en pleine évolution."
        } else if (serviceType.includes("5-6-7")) {
          message +=
            "\n\nNotre expertise dans les 5e, 6e et 7e arrondissements de Paris comprend une connaissance approfondie de ces quartiers historiques et prestigieux du centre de Paris."
        } else if (serviceType.includes("8-16-17")) {
          message +=
            "\n\nNotre expertise dans les 8e, 16e et 17e arrondissements de Paris comprend une connaissance approfondie de ces quartiers haut de gamme et résidentiels de l'ouest parisien."
        } else if (serviceType.includes("centre")) {
          message +=
            "\n\nNotre expertise dans Paris Centre (1er, 2e, 3e et 4e arrondissements) comprend une connaissance approfondie de ces quartiers historiques au cœur de la capitale."
        } else {
          message +=
            "\n\nNotre expertise à Paris comprend une connaissance approfondie de tous les arrondissements et de leurs spécificités."
        }
      }

      setTimeout(() => {
        addBotMessage(
          message,
          ["Tarifs et prestations", "Secteurs géographiques", "Prendre rendez-vous", "Autres services"],
          [
            {
              type: "service" as const,
              label: "En savoir plus",
              icon: <Building className="h-4 w-4" />,
              onClick: () => window.open("/services/administrateur-biens-paris", "_blank"),
            },
            {
              type: "appointment" as const,
              label: "Prendre RDV",
              icon: <Calendar className="h-4 w-4" />,
              onClick: () => showContactInformation(serviceType),
            },
          ],
        )
      }, 1500)
      return
    }

    if (serviceType === "garantie-loyers-impayes") {
      setTimeout(() => {
        addBotMessage(
          "Notre Garantie Loyers Impayés vous protège contre les risques locatifs : prise en charge des loyers impayés jusqu'à 100 000€, frais de contentieux, dégradations. Nos tarifs sont compétitifs à partir de 2,5% des loyers annuels.",
          ["Tarifs et conditions", "Comment ça marche", "Prendre rendez-vous", "Autres services"],
          [
            {
              type: "service" as const,
              label: "En savoir plus",
              icon: <Shield className="h-4 w-4" />,
              onClick: () => window.open("/services/garantie-loyers-impayes", "_blank"),
            },
            {
              type: "appointment" as const,
              label: "Prendre RDV",
              icon: <Calendar className="h-4 w-4" />,
              onClick: () => showContactInformation(serviceType),
            },
          ],
        )
      }, 1500)
      return
    }

    if (serviceType === "assurance-pno") {
      setTimeout(() => {
        addBotMessage(
          "L'Assurance Propriétaire Non Occupant protège votre bien : dégâts des eaux, incendie, vol, responsabilité civile. Couverture adaptée aux biens locatifs avec tarifs préférentiels à partir de 80€/an pour un studio.",
          ["Tarifs et conditions", "Comment ça marche", "Prendre rendez-vous", "Autres services"],
          [
            {
              type: "service" as const,
              label: "En savoir plus",
              icon: <Shield className="h-4 w-4" />,
              onClick: () => window.open("/services/assurance-pno", "_blank"),
            },
            {
              type: "appointment" as const,
              label: "Prendre RDV",
              icon: <Calendar className="h-4 w-4" />,
              onClick: () => showContactInformation(serviceType),
            },
          ],
        )
      }, 1500)
      return
    }

    if (serviceType === "comptabilite-mandants") {
      setTimeout(() => {
        addBotMessage(
          "Notre service Comptabilité Mandants assure la tenue complète de vos comptes immobiliers : suivi recettes/dépenses, décomptes annuels, déclarations fiscales, optimisation fiscale. Tarifs à partir de 25€/mois par lot.",
          ["Tarifs et conditions", "Comment ça marche", "Prendre rendez-vous", "Autres services"],
          [
            {
              type: "service" as const,
              label: "En savoir plus",
              icon: <FileText className="h-4 w-4" />,
              onClick: () => window.open("/services/comptabilite-mandants", "_blank"),
            },
            {
              type: "appointment" as const,
              label: "Prendre RDV",
              icon: <Calendar className="h-4 w-4" />,
              onClick: () => showContactInformation(serviceType),
            },
          ],
        )
      }, 1500)
      return
    }

    if (serviceType === "service-juridique") {
      setTimeout(() => {
        addBotMessage(
          "Notre service juridique vous accompagne : contentieux locatifs, procédures d'expulsion, recouvrement, conseils juridiques spécialisés. Tarifs sur devis selon la complexité du dossier.",
          ["Tarifs et conditions", "Comment ça marche", "Prendre rendez-vous", "Autres services"],
          [
            {
              type: "service" as const,
              label: "En savoir plus",
              icon: <Scale className="h-4 w-4" />,
              onClick: () => window.open("/services/juridique", "_blank"),
            },
            {
              type: "appointment" as const,
              label: "Prendre RDV",
              icon: <Calendar className="h-4 w-4" />,
              onClick: () => showContactInformation(serviceType),
            },
          ],
        )
      }, 1500)
      return
    }

    // Service par défaut
    setTimeout(() => {
      addBotMessage(
        "Nous proposons une gamme complète de services immobiliers. Quel service vous intéresse particulièrement ?",
        [
          "Administrateur de Biens",
          "Garantie Loyers Impayés",
          "Assurance PNO",
          "Comptabilité Mandants",
          "Service Juridique",
          "Syndic de Copropriété",
        ],
      )
    }, 1500)
  }

  const handleFlowProgression = (selectedOption: string): void => {
    const { profile, currentStep } = conversationState

    // Gestion spéciale pour "Prendre rendez-vous"
    if (
      ["Prendre rendez-vous", "Prendre RDV", "RDV expert", "Parler à un conseiller", "Contacter l'agence"].includes(
        selectedOption,
      )
    ) {
      showContactInformation(conversationState.serviceInterest)
      return
    }

    // Gestion des options de navigation principale
    if (["Acheter un bien", "Acheter"].includes(selectedOption)) {
      updateConversationState({ profile: "acheteur_primo", currentStep: "welcome" })
      const welcomeStep = getWelcomeMessage("acheteur_primo")
      setTimeout(() => {
        addBotMessage(welcomeStep.message, welcomeStep.options, getActionsForProfile("acheteur_primo"))
      }, 1000)
      return
    }

    if (["Vendre un bien", "Vendre"].includes(selectedOption)) {
      updateConversationState({ profile: "vendeur", currentStep: "welcome" })
      const welcomeStep = getWelcomeMessage("vendeur")
      setTimeout(() => {
        addBotMessage(welcomeStep.message, welcomeStep.options, getActionsForProfile("vendeur"))
      }, 1000)
      return
    }

    if (["Louer mon bien", "Gestion locative", "Mettre en location"].includes(selectedOption)) {
      updateConversationState({ profile: "proprietaire_bailleur", currentStep: "welcome" })
      const welcomeStep = getWelcomeMessage("proprietaire_bailleur")
      setTimeout(() => {
        addBotMessage(welcomeStep.message, welcomeStep.options, getActionsForProfile("proprietaire_bailleur"))
      }, 1000)
      return
    }

    if (["Chercher un logement", "Louer un logement"].includes(selectedOption)) {
      updateConversationState({ profile: "locataire", currentStep: "welcome" })
      const welcomeStep = getWelcomeMessage("locataire")
      setTimeout(() => {
        addBotMessage(welcomeStep.message, welcomeStep.options, getActionsForProfile("locataire"))
      }, 1000)
      return
    }

    if (["Investir"].includes(selectedOption)) {
      updateConversationState({ profile: "investisseur", currentStep: "welcome" })
      const welcomeStep = getWelcomeMessage("investisseur")
      setTimeout(() => {
        addBotMessage(welcomeStep.message, welcomeStep.options, getActionsForProfile("investisseur"))
      }, 1000)
      return
    }

    if (["Nos services", "Voir nos services", "Autres services", "Voir vos services"].includes(selectedOption)) {
      updateConversationState({ profile: "service_inquiry", currentStep: "welcome" })
      const welcomeStep = getWelcomeMessage("service_inquiry")
      setTimeout(() => {
        addBotMessage(welcomeStep.message, welcomeStep.options, getActionsForProfile("service_inquiry"))
      }, 1000)
      return
    }

    // Gestion des options de services spécifiques
    if (selectedOption === "Administrateur de Biens") {
      updateConversationState({
        profile: "service_inquiry",
        currentStep: "admin_biens_details",
        serviceInterest: "administrateur-biens-paris",
      })
      const step = conversationFlows.service_inquiry.admin_biens_details
      setTimeout(() => {
        addBotMessage(step.message, step.options, getActionsForProfile("service_inquiry", "administrateur-biens-paris"))
      }, 1000)
      return
    }

    if (selectedOption === "Garantie Loyers Impayés") {
      updateConversationState({
        profile: "service_inquiry",
        currentStep: "gli_details",
        serviceInterest: "garantie-loyers-impayes",
      })
      const step = conversationFlows.service_inquiry.gli_details
      setTimeout(() => {
        addBotMessage(step.message, step.options, getActionsForProfile("service_inquiry", "garantie-loyers-impayes"))
      }, 1000)
      return
    }

    if (selectedOption === "Assurance PNO") {
      updateConversationState({
        profile: "service_inquiry",
        currentStep: "pno_details",
        serviceInterest: "assurance-pno",
      })
      const step = conversationFlows.service_inquiry.pno_details
      setTimeout(() => {
        addBotMessage(step.message, step.options, getActionsForProfile("service_inquiry", "assurance-pno"))
      }, 1000)
      return
    }

    // Progression dans le flux actuel
    const currentFlow = conversationFlows[profile]
    const currentStepData = currentFlow?.[currentStep]

    if (!currentStepData) {
      // Étape non trouvée, revenir au début
      setTimeout(() => {
        addBotMessage("Je vais vous rediriger vers le bon service. Que souhaitez-vous faire ?", [
          "Acheter",
          "Vendre",
          "Louer",
          "Investir",
          "Nos services",
          "Parler à un conseiller",
        ])
      }, 1000)
      return
    }

    // Sauvegarder la qualification si applicable
    if (currentStepData.qualification) {
      const newQualificationData = { ...conversationState.qualificationData }
      newQualificationData[currentStepData.qualification] = selectedOption
      updateConversationState({ qualificationData: newQualificationData })
    }

    // Passer à l'étape suivante
    const nextStep = getNextStep(profile, currentStep, selectedOption)

    if (nextStep) {
      updateConversationState({ currentStep: nextStep.id })

      setTimeout(() => {
        const actions = nextStep.action ? getActionsForProfile(profile, conversationState.serviceInterest) : []
        addBotMessage(nextStep.message, nextStep.options, actions)

        // Exécuter l'action si définie
        if (nextStep.action) {
          executeAction(nextStep.action)
        }
      }, 1000)
    } else {
      // Fin du flux, proposer des actions finales et afficher les contacts
      setTimeout(() => {
        const finalMessage = getFinalMessage(profile)
        const finalActions = getActionsForProfile(profile, conversationState.serviceInterest)
        addBotMessage(
          finalMessage,
          ["Prendre rendez-vous", "Recevoir des propositions", "Autres questions"],
          finalActions,
        )

        // Afficher automatiquement les informations de contact après la fin du flux
        setTimeout(() => {
          showContactInformation(conversationState.serviceInterest)
        }, 2000)
      }, 1000)
    }
  }

  const executeAction = (action: string): void => {
    const serviceInterest = conversationState.serviceInterest

    switch (action) {
      case "show_admin_biens_paris":
        setTimeout(() => {
          addBotMessage(
            "Notre service d'Administration de Biens à Paris comprend : gestion locative premium, suivi technique personnalisé, valorisation patrimoniale, comptabilité dédiée. Nos honoraires sont compétitifs à partir de 4,5% HT des loyers annuels.",
            ["Tarifs détaillés", "Prendre rendez-vous", "Autres services"],
            [
              {
                type: "service" as const,
                label: "En savoir plus",
                icon: <Building className="h-4 w-4" />,
                onClick: () => window.open("/services/administrateur-biens-paris", "_blank"),
              },
            ],
          )
        }, 1000)
        break

      case "show_admin_biens_92":
      case "show_admin_biens_idf":
      case "show_admin_biens_haussmann":
      case "show_admin_biens_autre":
        setTimeout(() => {
          addBotMessage(
            "Notre service d'Administration de Biens s'adapte à votre secteur géographique avec des équipes dédiées et une connaissance approfondie du marché local. Souhaitez-vous être contacté pour un devis personnalisé ?",
            ["Oui, être contacté", "Plus d'informations", "Pas maintenant"],
            [
              {
                type: "service" as const,
                label: "En savoir plus",
                icon: <Building className="h-4 w-4" />,
                onClick: () => window.open("/services/administrateur-biens-ile-de-france", "_blank"),
              },
            ],
          )
        }, 1000)
        break

      case "show_gli_info":
        setTimeout(() => {
          addBotMessage(
            "Notre Garantie Loyers Impayés vous protège contre les risques locatifs : prise en charge des loyers impayés jusqu'à 100 000€, frais de contentieux, dégradations. Nos tarifs sont compétitifs à partir de 2,5% des loyers annuels.",
            ["Tarifs détaillés", "Prendre rendez-vous", "Autres services"],
            [
              {
                type: "service" as const,
                label: "En savoir plus",
                icon: <Shield className="h-4 w-4" />,
                onClick: () => window.open("/services/garantie-loyers-impayes", "_blank"),
              },
            ],
          )
        }, 1000)
        break

      case "show_pno_info":
        setTimeout(() => {
          addBotMessage(
            "L'Assurance Propriétaire Non Occupant protège votre bien : dégâts des eaux, incendie, vol, responsabilité civile. Couverture adaptée aux biens locatifs avec tarifs préférentiels à partir de 80€/an pour un studio.",
            ["Tarifs détaillés", "Prendre rendez-vous", "Autres services"],
            [
              {
                type: "service" as const,
                label: "En savoir plus",
                icon: <Shield className="h-4 w-4" />,
                onClick: () => window.open("/services/assurance-pno", "_blank"),
              },
            ],
          )
        }, 1000)
        break

      case "calculator":
        // Ouvrir le calculateur
        setTimeout(() => {
          addBotMessage("Voici nos calculateurs qui pourront vous aider dans votre projet :", [
            "Simuler un crédit",
            "Calculer une rentabilité",
            "Estimer des frais de notaire",
            "Prendre rendez-vous",
          ])
        }, 1000)
        break
      case "schedule_appointment":
        // Logique de prise de RDV
        showContactInformation(serviceInterest)
        break
      case "estimation_service":
        // Service d'estimation
        setTimeout(() => {
          addBotMessage(
            "Notre service d'estimation gratuite vous permet d'obtenir une valeur précise de votre bien. Souhaitez-vous être contacté par un expert ?",
            ["Oui, être contacté", "Plus d'informations", "Pas maintenant"],
          )
        }, 1000)
        break
      case "create_alert":
        // Créer une alerte
        setTimeout(() => {
          addBotMessage(
            "Je peux créer une alerte personnalisée pour vous informer des nouveaux biens correspondant à vos critères. Souhaitez-vous être contacté pour finaliser cette alerte ?",
            ["Oui, être contacté", "Plus d'informations", "Pas maintenant"],
          )

          // Proposer un contact après quelques secondes
          setTimeout(() => {
            showContactInformation(serviceInterest)
          }, 2000)
        }, 1000)
        break
      case "investment_tools":
        // Outils d'investissement
        setTimeout(() => {
          addBotMessage(
            "Nos outils d'investissement vous permettent de simuler la rentabilité de votre projet. Souhaitez-vous être contacté par un conseiller spécialisé ?",
            ["Oui, être contacté", "Voir les simulateurs", "Pas maintenant"],
          )
        }, 1000)
        break
      case "professional_service":
        // Service professionnel
        setTimeout(() => {
          addBotMessage(
            "Notre équipe dédiée aux professionnels est à votre disposition pour vous accompagner dans votre projet. Souhaitez-vous être contacté ?",
            ["Oui, être contacté", "Plus d'informations", "Pas maintenant"],
          )

          // Proposer un contact après quelques secondes
          setTimeout(() => {
            showContactInformation(serviceInterest)
          }, 2000)
        }, 1000)
        break
      case "gestion_locative_service":
        setTimeout(() => {
          addBotMessage(
            "Notre service de gestion locative comprend : recherche de locataires, encaissement des loyers, gestion des travaux, suivi administratif. Tarifs à partir de 3,9% HT. Souhaitez-vous être contacté pour un devis personnalisé ?",
            ["Oui, être contacté", "Plus d'informations", "Voir les tarifs"],
          )

          setTimeout(() => {
            showContactInformation(serviceInterest)
          }, 2000)
        }, 1000)
        break
    }
  }

  // Fonction pour afficher les informations de contact
  const showContactInformation = (serviceType?: string) => {
    // Déterminer le bon contact selon le service ou le profil
    let contactKey = conversationState.profile

    if (serviceType) {
      contactKey = serviceType
    }

    const contact = agencyContacts[contactKey] || agencyContacts.unknown

    // Message avec les informations de contact
    const contactMessage = `
    Voici les coordonnées de votre contact privilégié :
    
    👤 ${contact.name}
    🏢 ${contact.role}
    📍 92, Avenue de Saint-Mandé Paris 12e
    📞 ${contact.phone}
    ✉️ ${contact.email}
    
    Vous pouvez également nous contacter via notre formulaire en ligne ou directement à l'agence.
  `

    addBotMessage(
      contactMessage,
      ["Appeler maintenant", "Envoyer un email", "Formulaire de contact"],
      [
        {
          type: "contact",
          label: "Appeler",
          icon: <Phone className="h-4 w-4" />,
          onClick: () => window.open(`tel:${contact.phone.replace(/\s/g, "")}`, "_blank"),
        },
        {
          type: "contact",
          label: "Email",
          icon: <Mail className="h-4 w-4" />,
          onClick: () => window.open(`mailto:${contact.email}`, "_blank"),
        },
      ],
      true,
    )

    updateConversationState({ showContactInfo: true })
  }

  const getFinalMessage = (profile: UserProfile): string => {
    const messages = {
      acheteur_primo:
        "Parfait ! J'ai toutes les informations pour vous accompagner dans votre projet d'achat. Nos conseillers spécialisés primo-accédants vont pouvoir vous proposer des biens adaptés à vos critères.",
      vendeur:
        "Excellent ! Avec ces informations, nous pouvons vous proposer une estimation précise et un accompagnement personnalisé pour la vente de votre bien.",
      locataire:
        "Super ! J'ai bien noté vos critères de recherche. Nous allons pouvoir vous proposer des logements correspondant exactement à vos besoins.",
      investisseur:
        "Parfait ! Avec votre profil d'investisseur, nous pouvons vous présenter des opportunités adaptées à vos objectifs de rentabilité.",
      professionnel:
        "Excellent ! Notre équipe dédiée aux professionnels va pouvoir vous accompagner dans votre recherche de locaux.",
      service_inquiry:
        "Merci pour votre intérêt pour nos services. Un conseiller spécialisé va pouvoir vous donner toutes les informations détaillées et répondre à vos questions.",
      unknown:
        "Merci pour ces informations ! Un de nos conseillers va pouvoir vous accompagner selon vos besoins spécifiques.",
    }
    return messages[profile] || messages.unknown
  }

  const getActionsForProfile = (profile: UserProfile, serviceType?: string): ChatAction[] => {
    // Si on a un service spécifique, retourner les actions pour ce service
    if (serviceType) {
      if (serviceType.includes("administrateur-biens")) {
        return [
          {
            type: "service" as const,
            label: "En savoir plus",
            icon: <Building className="h-4 w-4" />,
            onClick: () => window.open("/services/administrateur-biens-paris", "_blank"),
          },
          {
            type: "appointment" as const,
            label: "Prendre RDV",
            icon: <Calendar className="h-4 w-4" />,
            onClick: () => showContactInformation(serviceType),
          },
        ]
      }

      if (serviceType === "garantie-loyers-impayes") {
        return [
          {
            type: "service" as const,
            label: "En savoir plus",
            icon: <Shield className="h-4 w-4" />,
            onClick: () => window.open("/services/garantie-loyers-impayes", "_blank"),
          },
          {
            type: "appointment" as const,
            label: "Prendre RDV",
            icon: <Calendar className="h-4 w-4" />,
            onClick: () => showContactInformation(serviceType),
          },
        ]
      }

      if (serviceType === "assurance-pno") {
        return [
          {
            type: "service" as const,
            label: "En savoir plus",
            icon: <Shield className="h-4 w-4" />,
            onClick: () => window.open("/services/assurance-pno", "_blank"),
          },
          {
            type: "appointment" as const,
            label: "Prendre RDV",
            icon: <Calendar className="h-4 w-4" />,
            onClick: () => showContactInformation(serviceType),
          },
        ]
      }
    }

    // Sinon, retourner les actions par défaut pour le profil
    const baseActions = {
      acheteur_primo: [
        {
          type: "calculator" as const,
          label: "Simuler mon crédit",
          icon: <Calculator className="h-4 w-4" />,
          onClick: () => window.open("/calculateurs", "_blank"),
        },
        {
          type: "appointment" as const,
          label: "Prendre RDV",
          icon: <Calendar className="h-4 w-4" />,
          onClick: () => showContactInformation(),
        },
      ],
      vendeur: [
        {
          type: "estimation" as const,
          label: "Estimation gratuite",
          icon: <Home className="h-4 w-4" />,
          onClick: () => executeAction("estimation_service"),
        },
        {
          type: "appointment" as const,
          label: "RDV expert",
          icon: <Calendar className="h-4 w-4" />,
          onClick: () => showContactInformation(),
        },
      ],
      locataire: [
        {
          type: "alert" as const,
          label: "Créer une alerte",
          icon: <MapPin className="h-4 w-4" />,
          onClick: () => executeAction("create_alert"),
        },
        {
          type: "appointment" as const,
          label: "Contacter l'agence",
          icon: <Calendar className="h-4 w-4" />,
          onClick: () => showContactInformation(),
        },
      ],
      investisseur: [
        {
          type: "simulation" as const,
          label: "Simuler rentabilité",
          icon: <TrendingUp className="h-4 w-4" />,
          onClick: () => window.open("/calculateurs", "_blank"),
        },
        {
          type: "appointment" as const,
          label: "Conseil investissement",
          icon: <Calendar className="h-4 w-4" />,
          onClick: () => showContactInformation(),
        },
      ],
      professionnel: [
        {
          type: "appointment" as const,
          label: "RDV équipe pro",
          icon: <Building className="h-4 w-4" />,
          onClick: () => showContactInformation(),
        },
      ],
      unknown: [
        {
          type: "appointment" as const,
          label: "Contacter l'agence",
          icon: <Calendar className="h-4 w-4" />,
          onClick: () => showContactInformation(),
        },
      ],
      proprietaire_bailleur: [
        {
          type: "estimation" as const,
          label: "Estimer le loyer",
          icon: <Home className="h-4 w-4" />,
          onClick: () => executeAction("estimation_service"),
        },
        {
          type: "appointment" as const,
          label: "RDV gestion locative",
          icon: <Calendar className="h-4 w-4" />,
          onClick: () => showContactInformation(),
        },
      ],
      service_inquiry: [
        {
          type: "appointment" as const,
          label: "Prendre RDV",
          icon: <Calendar className="h-4 w-4" />,
          onClick: () => showContactInformation(),
        },
      ],
    }

    return baseActions[profile] || []
  }

  const handleSendMessage = (messageText?: string) => {
    const text = messageText || input
    if (!text.trim()) return

    addUserMessage(text)
    setInput("")

    processConversationFlow(text)
  }

  const handleOptionClick = (option: string) => {
    addUserMessage(option)
    handleFlowProgression(option)
  }

  const getProfileIcon = (profile: UserProfile) => {
    const icons = {
      acheteur_primo: "🏠",
      vendeur: "🏡",
      locataire: "🔑",
      proprietaire_bailleur: "🏢",
      investisseur: "📈",
      professionnel: "🏢",
      unknown: "❓",
      service_inquiry: "ℹ️",
    }
    return icons[profile]
  }

  const getQualificationSummary = (): string => {
    const { qualificationData } = conversationState
    const items = Object.entries(qualificationData)
      .map(([key, value]) => `${key}: ${value}`)
      .join(", ")
    return items || "En cours de qualification..."
  }

  return (
    <div
      className="fixed bottom-4 right-4 z-[9999]"
      style={{
        position: "fixed",
        bottom: "1rem",
        right: "1rem",
        zIndex: 9999,
        pointerEvents: "auto",
      }}
    >
      {isOpen ? (
        <div className="bg-white rounded-lg shadow-2xl w-96 h-[600px] flex flex-col border border-gray-200 transition-all duration-300 ease-in-out">
          {/* Header */}
          <div className="p-4 bg-gradient-to-r from-[#002395] to-[#001875] text-white rounded-t-lg flex justify-between items-center">
            <div className="flex items-center gap-2">
              <h3 className="font-semibold">Assistant DOGES</h3>
              {conversationState.profile !== "unknown" && (
                <span className="text-xs bg-white/20 px-2 py-1 rounded">
                  {getProfileIcon(conversationState.profile)} {conversationState.profile.replace("_", " ")}
                </span>
              )}
            </div>
            <Button
              variant="ghost"
              size="sm"
              className="text-white hover:text-white/80 hover:bg-white/10"
              onClick={() => setIsOpen(false)}
            >
              <X className="h-4 w-4" />
            </Button>
          </div>

          {/* Messages */}
          <div className="flex-1 overflow-y-auto p-4 space-y-4 bg-gray-50">
            {messages.map((message) => (
              <div key={message.id} className={`flex ${message.sender === "user" ? "justify-end" : "justify-start"}`}>
                <div className="max-w-[85%]">
                  <div
                    className={`p-3 rounded-lg ${
                      message.sender === "user"
                        ? "bg-[#002395] text-white"
                        : message.isContactInfo
                          ? "bg-gradient-to-r from-[#002395]/10 to-[#001875]/10 text-gray-800 border border-[#002395]/20"
                          : "bg-white text-gray-800 border border-gray-200"
                    }`}
                  >
                    {message.text}
                  </div>

                  {/* Options */}
                  {message.options && message.sender === "bot" && (
                    <div className="mt-2 space-y-1">
                      {message.options.map((option, index) => (
                        <button
                          key={index}
                          onClick={() => handleOptionClick(option)}
                          className="block w-full text-left p-2 text-sm bg-white border border-gray-200 rounded hover:bg-gray-50 hover:border-[#002395] transition-colors"
                        >
                          {option}
                        </button>
                      ))}
                    </div>
                  )}

                  {/* Actions */}
                  {message.actions && message.sender === "bot" && (
                    <div className="mt-2 flex flex-wrap gap-2">
                      {message.actions.map((action, index) => (
                        <button
                          key={index}
                          onClick={action.onClick}
                          className="flex items-center gap-1 px-3 py-1 text-xs bg-[#002395] text-white rounded hover:bg-[#001875] transition-colors"
                        >
                          {action.icon}
                          {action.label}
                        </button>
                      ))}
                    </div>
                  )}

                  <div className="text-xs text-gray-500 mt-1">
                    {message.timestamp.toLocaleTimeString([], { hour: "2-digit", minute: "2-digit" })}
                  </div>
                </div>
              </div>
            ))}

            {isTyping && (
              <div className="flex justify-start">
                <div className="bg-white p-3 rounded-lg border border-gray-200">
                  <div className="flex space-x-1">
                    <div className="w-2 h-2 bg-gray-400 rounded-full animate-bounce"></div>
                    <div
                      className="w-2 h-2 bg-gray-400 rounded-full animate-bounce"
                      style={{ animationDelay: "0.1s" }}
                    ></div>
                    <div
                      className="w-2 h-2 bg-gray-400 rounded-full animate-bounce"
                      style={{ animationDelay: "0.2s" }}
                    ></div>
                  </div>
                </div>
              </div>
            )}
            <div ref={messagesEndRef} />
          </div>

          {/* Input */}
          <form
            onSubmit={(e) => {
              e.preventDefault()
              handleSendMessage()
            }}
            className="p-4 border-t bg-white rounded-b-lg"
          >
            <div className="flex gap-2">
              <Input
                value={input}
                onChange={(e) => setInput(e.target.value)}
                placeholder="Tapez votre message..."
                className="flex-1"
                disabled={isTyping}
              />
              <Button
                type="submit"
                size="icon"
                className="bg-[#002395] hover:bg-[#001875] text-white"
                disabled={isTyping || !input.trim()}
              >
                <Send className="h-4 w-4" />
              </Button>
            </div>

            {/* Qualification Status */}
            {conversationState.profile !== "unknown" && Object.keys(conversationState.qualificationData).length > 0 && (
              <div className="mt-2 text-xs text-gray-500">
                <div className="font-medium">Profil: {conversationState.profile.replace("_", " ")}</div>
                <div>Critères: {getQualificationSummary()}</div>
              </div>
            )}
          </form>
        </div>
      ) : (
        <Button
          onClick={() => setIsOpen(true)}
          className={`rounded-full w-14 h-14 bg-gradient-to-r from-[#002395] to-[#001875] hover:from-[#001875] hover:to-[#002395] shadow-lg hover:shadow-xl transition-all duration-500 flex items-center justify-center group ${
            hasNewMessage ? "animate-subtle-pulse" : ""
          }`}
          aria-label="Ouvrir l'assistant intelligent"
          style={{
            position: "relative",
            zIndex: 9999,
          }}
        >
          <MessageCircle className="h-6 w-6 text-white group-hover:scale-110 transition-transform" />
          {hasNewMessage && (
            <div className="absolute -top-1 -right-1 w-3 h-3 bg-red-500 rounded-full transition-opacity duration-1000 opacity-75"></div>
          )}
        </Button>
      )}
    </div>
  )
}
