"use client"

import type React from "react"

import { useState } from "react"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Button } from "@/components/ui/button"
import { Checkbox } from "@/components/ui/checkbox"
import {
  CheckCircle,
  AlertTriangle,
  Globe,
  Smartphone,
  Search,
  Shield,
  Zap,
  FileText,
  Users,
  Settings,
  BarChart3,
} from "lucide-react"

interface TestItem {
  id: string
  title: string
  description: string
  priority: "critical" | "high" | "medium" | "low"
  category: string
  tested: boolean
  notes?: string
}

interface TestCategory {
  name: string
  icon: React.ReactNode
  color: string
  items: TestItem[]
}

export default function PreDeploymentChecklist() {
  const [testCategories, setTestCategories] = useState<TestCategory[]>([
    {
      name: "Tests Fonctionnels",
      icon: <Settings className="h-5 w-5" />,
      color: "blue",
      items: [
        {
          id: "nav-main",
          title: "Navigation principale",
          description: "Tous les liens du menu principal fonctionnent",
          priority: "critical",
          category: "navigation",
          tested: false,
        },
        {
          id: "nav-footer",
          title: "Navigation footer",
          description: "Tous les liens du footer sont fonctionnels",
          priority: "high",
          category: "navigation",
          tested: false,
        },
        {
          id: "breadcrumbs",
          title: "Fils d'Ariane",
          description: "Breadcrumbs affichés et fonctionnels sur toutes les pages",
          priority: "high",
          category: "navigation",
          tested: false,
        },
        {
          id: "contact-forms",
          title: "Formulaires de contact",
          description: "Tous les formulaires envoient correctement les données",
          priority: "critical",
          category: "forms",
          tested: false,
        },
        {
          id: "calculators",
          title: "Calculateurs",
          description: "Calculateurs de rentabilité et frais fonctionnent",
          priority: "high",
          category: "tools",
          tested: false,
        },
        {
          id: "search",
          title: "Recherche blog",
          description: "Fonction de recherche dans le blog opérationnelle",
          priority: "medium",
          category: "search",
          tested: false,
        },
      ],
    },
    {
      name: "Tests de Contenu",
      icon: <FileText className="h-5 w-5" />,
      color: "green",
      items: [
        {
          id: "content-spelling",
          title: "Orthographe et grammaire",
          description: "Vérification complète de tous les textes",
          priority: "high",
          category: "content",
          tested: false,
        },
        {
          id: "content-images",
          title: "Images et médias",
          description: "Toutes les images s'affichent correctement",
          priority: "critical",
          category: "media",
          tested: false,
        },
        {
          id: "content-links",
          title: "Liens internes/externes",
          description: "Aucun lien brisé sur le site",
          priority: "critical",
          category: "links",
          tested: false,
        },
        {
          id: "content-contact",
          title: "Informations de contact",
          description: "Adresse, téléphone, email corrects partout",
          priority: "critical",
          category: "contact",
          tested: false,
        },
        {
          id: "content-legal",
          title: "Pages légales",
          description: "Mentions légales, CGV, politique de confidentialité",
          priority: "high",
          category: "legal",
          tested: false,
        },
      ],
    },
    {
      name: "Tests Techniques",
      icon: <Zap className="h-5 w-5" />,
      color: "purple",
      items: [
        {
          id: "tech-404",
          title: "Pages d'erreur",
          description: "Pages 404, 500 personnalisées fonctionnent",
          priority: "high",
          category: "errors",
          tested: false,
        },
        {
          id: "tech-redirects",
          title: "Redirections",
          description: "Toutes les redirections fonctionnent correctement",
          priority: "high",
          category: "redirects",
          tested: false,
        },
        {
          id: "tech-sitemap",
          title: "Sitemap XML",
          description: "Sitemap généré et accessible",
          priority: "high",
          category: "seo",
          tested: false,
        },
        {
          id: "tech-robots",
          title: "Robots.txt",
          description: "Fichier robots.txt configuré correctement",
          priority: "medium",
          category: "seo",
          tested: false,
        },
        {
          id: "tech-ssl",
          title: "Certificat SSL",
          description: "HTTPS fonctionnel sur tout le site",
          priority: "critical",
          category: "security",
          tested: false,
        },
        {
          id: "tech-performance",
          title: "Performance",
          description: "Temps de chargement < 3s sur toutes les pages",
          priority: "high",
          category: "performance",
          tested: false,
        },
      ],
    },
    {
      name: "Tests SEO",
      icon: <Search className="h-5 w-5" />,
      color: "yellow",
      items: [
        {
          id: "seo-titles",
          title: "Titres de pages",
          description: "Tous les titres sont uniques et optimisés",
          priority: "critical",
          category: "seo",
          tested: false,
        },
        {
          id: "seo-meta",
          title: "Meta descriptions",
          description: "Meta descriptions présentes et optimisées",
          priority: "high",
          category: "seo",
          tested: false,
        },
        {
          id: "seo-h1",
          title: "Balises H1",
          description: "Une seule H1 par page, bien structurée",
          priority: "high",
          category: "seo",
          tested: false,
        },
        {
          id: "seo-alt",
          title: "Attributs alt",
          description: "Toutes les images ont des attributs alt descriptifs",
          priority: "medium",
          category: "seo",
          tested: false,
        },
        {
          id: "seo-schema",
          title: "Schema markup",
          description: "Données structurées implémentées",
          priority: "medium",
          category: "seo",
          tested: false,
        },
        {
          id: "seo-google",
          title: "Google Search Console",
          description: "Site vérifié et indexé par Google",
          priority: "high",
          category: "seo",
          tested: false,
        },
      ],
    },
    {
      name: "Tests Responsive",
      icon: <Smartphone className="h-5 w-5" />,
      color: "indigo",
      items: [
        {
          id: "responsive-mobile",
          title: "Mobile (320px-768px)",
          description: "Affichage parfait sur smartphones",
          priority: "critical",
          category: "responsive",
          tested: false,
        },
        {
          id: "responsive-tablet",
          title: "Tablette (768px-1024px)",
          description: "Affichage optimisé sur tablettes",
          priority: "high",
          category: "responsive",
          tested: false,
        },
        {
          id: "responsive-desktop",
          title: "Desktop (1024px+)",
          description: "Affichage parfait sur ordinateurs",
          priority: "high",
          category: "responsive",
          tested: false,
        },
        {
          id: "responsive-touch",
          title: "Interactions tactiles",
          description: "Boutons et liens facilement cliquables",
          priority: "high",
          category: "responsive",
          tested: false,
        },
        {
          id: "responsive-orientation",
          title: "Orientation",
          description: "Fonctionne en portrait et paysage",
          priority: "medium",
          category: "responsive",
          tested: false,
        },
      ],
    },
    {
      name: "Tests Navigateurs",
      icon: <Globe className="h-5 w-5" />,
      color: "red",
      items: [
        {
          id: "browser-chrome",
          title: "Google Chrome",
          description: "Fonctionnement parfait sur Chrome",
          priority: "critical",
          category: "browsers",
          tested: false,
        },
        {
          id: "browser-firefox",
          title: "Mozilla Firefox",
          description: "Fonctionnement parfait sur Firefox",
          priority: "high",
          category: "browsers",
          tested: false,
        },
        {
          id: "browser-safari",
          title: "Safari",
          description: "Fonctionnement parfait sur Safari",
          priority: "high",
          category: "browsers",
          tested: false,
        },
        {
          id: "browser-edge",
          title: "Microsoft Edge",
          description: "Fonctionnement parfait sur Edge",
          priority: "medium",
          category: "browsers",
          tested: false,
        },
        {
          id: "browser-mobile",
          title: "Navigateurs mobiles",
          description: "Chrome Mobile, Safari Mobile, Samsung Internet",
          priority: "high",
          category: "browsers",
          tested: false,
        },
      ],
    },
    {
      name: "Tests Accessibilité",
      icon: <Users className="h-5 w-5" />,
      color: "pink",
      items: [
        {
          id: "a11y-keyboard",
          title: "Navigation clavier",
          description: "Site entièrement navigable au clavier",
          priority: "high",
          category: "accessibility",
          tested: false,
        },
        {
          id: "a11y-screen-reader",
          title: "Lecteurs d'écran",
          description: "Compatible avec NVDA, JAWS, VoiceOver",
          priority: "high",
          category: "accessibility",
          tested: false,
        },
        {
          id: "a11y-contrast",
          title: "Contraste des couleurs",
          description: "Ratio de contraste conforme WCAG 2.1",
          priority: "medium",
          category: "accessibility",
          tested: false,
        },
        {
          id: "a11y-focus",
          title: "Indicateurs de focus",
          description: "Focus visible sur tous les éléments interactifs",
          priority: "medium",
          category: "accessibility",
          tested: false,
        },
      ],
    },
    {
      name: "Tests Sécurité",
      icon: <Shield className="h-5 w-5" />,
      color: "gray",
      items: [
        {
          id: "security-forms",
          title: "Sécurité des formulaires",
          description: "Protection contre spam et injections",
          priority: "critical",
          category: "security",
          tested: false,
        },
        {
          id: "security-headers",
          title: "Headers de sécurité",
          description: "CSP, HSTS, X-Frame-Options configurés",
          priority: "high",
          category: "security",
          tested: false,
        },
        {
          id: "security-dependencies",
          title: "Dépendances",
          description: "Aucune vulnérabilité dans les packages",
          priority: "high",
          category: "security",
          tested: false,
        },
        {
          id: "security-data",
          title: "Protection des données",
          description: "Conformité RGPD et protection des données",
          priority: "critical",
          category: "security",
          tested: false,
        },
      ],
    },
    {
      name: "Tests Analytics",
      icon: <BarChart3 className="h-5 w-5" />,
      color: "orange",
      items: [
        {
          id: "analytics-ga",
          title: "Google Analytics",
          description: "Tracking correctement configuré",
          priority: "high",
          category: "analytics",
          tested: false,
        },
        {
          id: "analytics-gtm",
          title: "Google Tag Manager",
          description: "GTM installé et fonctionnel",
          priority: "medium",
          category: "analytics",
          tested: false,
        },
        {
          id: "analytics-events",
          title: "Événements personnalisés",
          description: "Tracking des conversions et interactions",
          priority: "medium",
          category: "analytics",
          tested: false,
        },
        {
          id: "analytics-goals",
          title: "Objectifs de conversion",
          description: "Objectifs configurés dans GA",
          priority: "medium",
          category: "analytics",
          tested: false,
        },
      ],
    },
  ])

  const toggleTest = (categoryIndex: number, itemIndex: number) => {
    const newCategories = [...testCategories]
    newCategories[categoryIndex].items[itemIndex].tested = !newCategories[categoryIndex].items[itemIndex].tested
    setTestCategories(newCategories)
  }

  const getProgress = () => {
    const totalTests = testCategories.reduce((acc, cat) => acc + cat.items.length, 0)
    const completedTests = testCategories.reduce((acc, cat) => acc + cat.items.filter((item) => item.tested).length, 0)
    return { total: totalTests, completed: completedTests, percentage: Math.round((completedTests / totalTests) * 100) }
  }

  const getCriticalIssues = () => {
    return testCategories.flatMap((cat) => cat.items).filter((item) => item.priority === "critical" && !item.tested)
  }

  const progress = getProgress()
  const criticalIssues = getCriticalIssues()

  const getPriorityColor = (priority: string) => {
    switch (priority) {
      case "critical":
        return "text-red-600 bg-red-100"
      case "high":
        return "text-orange-600 bg-orange-100"
      case "medium":
        return "text-yellow-600 bg-yellow-100"
      case "low":
        return "text-green-600 bg-green-100"
      default:
        return "text-gray-600 bg-gray-100"
    }
  }

  return (
    <div className="max-w-7xl mx-auto p-6 space-y-6">
      {/* Header */}
      <div className="text-center space-y-4">
        <h1 className="text-3xl font-bold bg-gradient-to-r from-[#002395] to-[#001875] bg-clip-text text-transparent">
          Checklist de Pré-Déploiement DOGES
        </h1>
        <p className="text-gray-600 max-w-3xl mx-auto">
          Tests complets à effectuer avant la mise en production du site web
        </p>
      </div>

      {/* Progress Overview */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center justify-between">
            <span>Progression Globale</span>
            <span className="text-2xl font-bold text-[#002395]">{progress.percentage}%</span>
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="space-y-4">
            <div className="w-full bg-gray-200 rounded-full h-3">
              <div
                className="bg-gradient-to-r from-[#002395] to-[#001875] h-3 rounded-full transition-all duration-300"
                style={{ width: `${progress.percentage}%` }}
              ></div>
            </div>
            <div className="flex justify-between text-sm text-gray-600">
              <span>
                {progress.completed} / {progress.total} tests complétés
              </span>
              <span>{progress.total - progress.completed} tests restants</span>
            </div>

            {criticalIssues.length > 0 && (
              <div className="bg-red-50 border border-red-200 rounded-lg p-4">
                <div className="flex items-center space-x-2 mb-2">
                  <AlertTriangle className="h-5 w-5 text-red-600" />
                  <span className="font-semibold text-red-800">Tests Critiques Non Complétés</span>
                </div>
                <ul className="space-y-1">
                  {criticalIssues.map((issue) => (
                    <li key={issue.id} className="text-sm text-red-700">
                      • {issue.title}
                    </li>
                  ))}
                </ul>
              </div>
            )}
          </div>
        </CardContent>
      </Card>

      {/* Test Categories */}
      <div className="grid lg:grid-cols-2 gap-6">
        {testCategories.map((category, categoryIndex) => {
          const categoryProgress = {
            total: category.items.length,
            completed: category.items.filter((item) => item.tested).length,
          }
          const categoryPercentage = Math.round((categoryProgress.completed / categoryProgress.total) * 100)

          return (
            <Card key={categoryIndex} className="h-fit">
              <CardHeader>
                <CardTitle className="flex items-center justify-between">
                  <div className="flex items-center space-x-2">
                    {category.icon}
                    <span>{category.name}</span>
                  </div>
                  <div className="flex items-center space-x-2">
                    <span className="text-sm text-gray-500">
                      {categoryProgress.completed}/{categoryProgress.total}
                    </span>
                    <div className="w-16 bg-gray-200 rounded-full h-2">
                      <div
                        className="bg-[#002395] h-2 rounded-full transition-all duration-300"
                        style={{ width: `${categoryPercentage}%` }}
                      ></div>
                    </div>
                  </div>
                </CardTitle>
              </CardHeader>
              <CardContent>
                <div className="space-y-3">
                  {category.items.map((item, itemIndex) => (
                    <div
                      key={item.id}
                      className={`flex items-start space-x-3 p-3 rounded-lg border transition-all duration-200 ${
                        item.tested ? "bg-green-50 border-green-200" : "bg-gray-50 border-gray-200"
                      }`}
                    >
                      <Checkbox
                        checked={item.tested}
                        onCheckedChange={() => toggleTest(categoryIndex, itemIndex)}
                        className="mt-1"
                      />
                      <div className="flex-1 space-y-1">
                        <div className="flex items-center justify-between">
                          <h4 className={`font-medium ${item.tested ? "line-through text-gray-500" : ""}`}>
                            {item.title}
                          </h4>
                          <span
                            className={`px-2 py-1 rounded-full text-xs font-medium ${getPriorityColor(item.priority)}`}
                          >
                            {item.priority}
                          </span>
                        </div>
                        <p className={`text-sm ${item.tested ? "line-through text-gray-400" : "text-gray-600"}`}>
                          {item.description}
                        </p>
                      </div>
                    </div>
                  ))}
                </div>
              </CardContent>
            </Card>
          )
        })}
      </div>

      {/* Final Deployment Checklist */}
      {progress.percentage === 100 && (
        <Card className="bg-green-50 border-green-200">
          <CardHeader>
            <CardTitle className="flex items-center space-x-2 text-green-800">
              <CheckCircle className="h-6 w-6" />
              <span>Prêt pour le Déploiement !</span>
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-4">
              <p className="text-green-700">
                Tous les tests ont été complétés avec succès. Le site DOGES est prêt pour la mise en production.
              </p>
              <div className="bg-white rounded-lg p-4 space-y-2">
                <h4 className="font-semibold text-green-800">Étapes finales avant déploiement :</h4>
                <ul className="space-y-1 text-sm text-green-700">
                  <li>✅ Sauvegarde complète du site actuel</li>
                  <li>✅ Configuration des DNS pour le domaine de production</li>
                  <li>✅ Mise en place du monitoring de performance</li>
                  <li>✅ Configuration des alertes d'erreur</li>
                  <li>✅ Test final sur l'environnement de production</li>
                  <li>✅ Communication au client de la mise en ligne</li>
                </ul>
              </div>
            </div>
          </CardContent>
        </Card>
      )}

      {/* Quick Actions */}
      <div className="flex justify-center space-x-4">
        <Button
          onClick={() => {
            const newCategories = testCategories.map((cat) => ({
              ...cat,
              items: cat.items.map((item) => ({ ...item, tested: true })),
            }))
            setTestCategories(newCategories)
          }}
          variant="outline"
        >
          Marquer tout comme testé
        </Button>
        <Button
          onClick={() => {
            const newCategories = testCategories.map((cat) => ({
              ...cat,
              items: cat.items.map((item) => ({ ...item, tested: false })),
            }))
            setTestCategories(newCategories)
          }}
          variant="outline"
        >
          Réinitialiser
        </Button>
      </div>
    </div>
  )
}

export { PreDeploymentChecklist }
