"use client"

import { useState } from "react"
import { CheckCircle, AlertCircle, XCircle } from "lucide-react"

interface ChecklistItem {
  id: string
  category: string
  item: string
  status: "completed" | "warning" | "pending"
  description: string
}

const checklistItems: ChecklistItem[] = [
  // Informations légales
  {
    id: "legal-1",
    category: "Informations Légales",
    item: "Vérifier SIRET et RCS",
    status: "warning",
    description: "SIRET: 949445571 - À confirmer avec le client",
  },
  {
    id: "legal-2",
    category: "Informations Légales",
    item: "Carte professionnelle",
    status: "warning",
    description: "CPI 7501 2023 000000431 - À vérifier",
  },
  {
    id: "legal-3",
    category: "Informations Légales",
    item: "Assurance RCP",
    status: "warning",
    description: "MMA A148537109 - À confirmer",
  },

  // Contact
  {
    id: "contact-1",
    category: "Contact",
    item: "Adresse physique",
    status: "warning",
    description: "92, Avenue de Saint-Mandé 75012 PARIS - À confirmer",
  },
  {
    id: "contact-2",
    category: "Contact",
    item: "Numéro de téléphone",
    status: "warning",
    description: "01 84 80 98 10 - À tester",
  },
  {
    id: "contact-3",
    category: "Contact",
    item: "Email principal",
    status: "completed",
    description: "contact@dogesadb.fr - OK",
  },

  // Technique
  {
    id: "tech-1",
    category: "Technique",
    item: "Domaine principal",
    status: "warning",
    description: "doges.fr vs dogesadb.fr - À clarifier",
  },
  {
    id: "tech-2",
    category: "Technique",
    item: "Certificat SSL",
    status: "pending",
    description: "À configurer lors du déploiement",
  },
  {
    id: "tech-3",
    category: "Technique",
    item: "Analytics/Tracking",
    status: "pending",
    description: "Google Analytics à configurer",
  },

  // SEO
  {
    id: "seo-1",
    category: "SEO",
    item: "Métadonnées",
    status: "completed",
    description: "Toutes les pages ont des métadonnées complètes",
  },
  {
    id: "seo-2",
    category: "SEO",
    item: "Sitemap",
    status: "completed",
    description: "Sitemap.xml généré automatiquement",
  },
  {
    id: "seo-3",
    category: "SEO",
    item: "Robots.txt",
    status: "completed",
    description: "Robots.txt configuré",
  },

  // Contenu
  {
    id: "content-1",
    category: "Contenu",
    item: "Tarifs uniformisés",
    status: "completed",
    description: "2% / 3.9% / 6.9% sur tous les calculateurs",
  },
  {
    id: "content-2",
    category: "Contenu",
    item: "Images optimisées",
    status: "completed",
    description: "Toutes les images ont des alt text et sont optimisées",
  },
  {
    id: "content-3",
    category: "Contenu",
    item: "Liens internes",
    status: "completed",
    description: "Navigation cohérente et liens fonctionnels",
  },
]

export default function PreLaunchChecklist() {
  const [checkedItems, setCheckedItems] = useState<Set<string>>(new Set())

  const toggleItem = (id: string) => {
    const newChecked = new Set(checkedItems)
    if (newChecked.has(id)) {
      newChecked.delete(id)
    } else {
      newChecked.add(id)
    }
    setCheckedItems(newChecked)
  }

  const getStatusIcon = (status: string) => {
    switch (status) {
      case "completed":
        return <CheckCircle className="h-5 w-5 text-green-500" />
      case "warning":
        return <AlertCircle className="h-5 w-5 text-yellow-500" />
      case "pending":
        return <XCircle className="h-5 w-5 text-red-500" />
      default:
        return <XCircle className="h-5 w-5 text-gray-400" />
    }
  }

  const categories = [...new Set(checklistItems.map((item) => item.category))]

  const completedCount = checklistItems.filter((item) => item.status === "completed").length
  const warningCount = checklistItems.filter((item) => item.status === "warning").length
  const pendingCount = checklistItems.filter((item) => item.status === "pending").length

  return (
    <div className="max-w-4xl mx-auto p-6">
      <div className="bg-white rounded-lg shadow-lg p-6">
        <h1 className="text-3xl font-bold mb-6 text-[#002395]">Checklist Pré-Lancement - Site DOGES</h1>

        {/* Résumé */}
        <div className="grid grid-cols-1 md:grid-cols-3 gap-4 mb-8">
          <div className="bg-green-50 p-4 rounded-lg border border-green-200">
            <div className="flex items-center">
              <CheckCircle className="h-6 w-6 text-green-500 mr-2" />
              <div>
                <div className="font-semibold text-green-800">{completedCount} Complétés</div>
                <div className="text-sm text-green-600">Prêts pour la mise en ligne</div>
              </div>
            </div>
          </div>

          <div className="bg-yellow-50 p-4 rounded-lg border border-yellow-200">
            <div className="flex items-center">
              <AlertCircle className="h-6 w-6 text-yellow-500 mr-2" />
              <div>
                <div className="font-semibold text-yellow-800">{warningCount} À Vérifier</div>
                <div className="text-sm text-yellow-600">Nécessitent validation client</div>
              </div>
            </div>
          </div>

          <div className="bg-red-50 p-4 rounded-lg border border-red-200">
            <div className="flex items-center">
              <XCircle className="h-6 w-6 text-red-500 mr-2" />
              <div>
                <div className="font-semibold text-red-800">{pendingCount} En Attente</div>
                <div className="text-sm text-red-600">À configurer au déploiement</div>
              </div>
            </div>
          </div>
        </div>

        {/* Checklist par catégorie */}
        {categories.map((category) => (
          <div key={category} className="mb-8">
            <h2 className="text-xl font-semibold mb-4 text-gray-800 border-b pb-2">{category}</h2>
            <div className="space-y-3">
              {checklistItems
                .filter((item) => item.category === category)
                .map((item) => (
                  <div
                    key={item.id}
                    className="flex items-start p-4 bg-gray-50 rounded-lg hover:bg-gray-100 transition-colors"
                  >
                    <div className="mr-3 mt-0.5">{getStatusIcon(item.status)}</div>
                    <div className="flex-1">
                      <div className="font-medium text-gray-900">{item.item}</div>
                      <div className="text-sm text-gray-600 mt-1">{item.description}</div>
                    </div>
                    <input
                      type="checkbox"
                      checked={checkedItems.has(item.id)}
                      onChange={() => toggleItem(item.id)}
                      className="ml-4 h-4 w-4 text-[#002395] rounded"
                    />
                  </div>
                ))}
            </div>
          </div>
        ))}

        {/* Recommandations */}
        <div className="mt-8 p-6 bg-blue-50 rounded-lg border border-blue-200">
          <h3 className="text-lg font-semibold text-blue-800 mb-3">Recommandations avant envoi client</h3>
          <ul className="space-y-2 text-blue-700">
            <li>• Demander confirmation de toutes les informations légales</li>
            <li>• Tester le numéro de téléphone et le lien Ringover</li>
            <li>• Clarifier le domaine définitif (doges.fr ou dogesadb.fr)</li>
            <li>• Préparer les accès Google Analytics</li>
            <li>• Planifier la configuration SSL</li>
          </ul>
        </div>

        {/* Status global */}
        <div className="mt-6 p-4 bg-gray-100 rounded-lg">
          <div className="text-center">
            <div className="text-lg font-semibold text-gray-800">
              Status Global: {warningCount === 0 && pendingCount === 0 ? "✅ PRÊT" : "⚠️ VALIDATION REQUISE"}
            </div>
            <div className="text-sm text-gray-600 mt-1">
              {warningCount > 0 || pendingCount > 0
                ? "Quelques points nécessitent une validation avant la mise en ligne"
                : "Le site est prêt pour la mise en ligne !"}
            </div>
          </div>
        </div>
      </div>
    </div>
  )
}
