"use client"

import { useState } from "react"
import Image from "next/image"
import { Button } from "@/components/ui/button"
import { Heart, Share2, MapPin } from "lucide-react"

export type PropertyCardProps = {
  id: string
  title: string
  location: string
  price: string
  surface: string
  rooms: string
  type: string
  image: string
  isFavorite?: boolean
}

export default function PropertyCard({
  title,
  location,
  price,
  surface,
  rooms,
  type,
  image,
  isFavorite: initialFavorite = false,
}: PropertyCardProps) {
  const [isFavorite, setIsFavorite] = useState(initialFavorite)

  const toggleFavorite = () => {
    setIsFavorite(!isFavorite)
  }

  const shareProperty = () => {
    if (navigator.share) {
      navigator.share({
        title,
        text: `Découvrez ce bien : ${title}`,
        url: window.location.href,
      })
    }
  }

  return (
    <div className="bg-white rounded-2xl overflow-hidden shadow-lg hover:shadow-2xl transition-all duration-500 group">
      <div className="relative">
        <div className="relative h-64">
          <Image src={image || "/placeholder.svg"} alt={title} fill className="object-cover" />
        </div>
        <div className="absolute top-4 right-4 flex gap-2">
          <Button
            variant="ghost"
            size="icon"
            className="bg-white/90 backdrop-blur-sm hover:bg-white"
            onClick={toggleFavorite}
          >
            <Heart className={`h-5 w-5 ${isFavorite ? "fill-red-500 text-red-500" : "text-gray-600"}`} />
          </Button>
          <Button
            variant="ghost"
            size="icon"
            className="bg-white/90 backdrop-blur-sm hover:bg-white"
            onClick={shareProperty}
          >
            <Share2 className="h-5 w-5 text-gray-600" />
          </Button>
        </div>
        <div className="absolute bottom-4 left-4 bg-white/90 backdrop-blur-sm px-3 py-1 rounded-full">
          <span className="text-[#002395] font-semibold">{price}</span>
        </div>
      </div>
      <div className="p-6">
        <h3 className="text-xl font-semibold mb-2">{title}</h3>
        <div className="flex items-center text-gray-600 mb-4">
          <MapPin className="h-4 w-4 mr-1" />
          {location}
        </div>
        <div className="flex items-center justify-between text-sm text-gray-500 mb-4">
          <span>{surface}</span>
          <span>{rooms}</span>
          <span>{type}</span>
        </div>
        <Button className="w-full bg-gradient-to-r from-[#002395] to-[#001875]">Contacter</Button>
      </div>
    </div>
  )
}
