"use client"

import { motion } from "framer-motion"
import Image from "next/image"
import Link from "next/link"
import { ArrowRight } from "lucide-react"
import { Button } from "@/components/ui/button"
import { memo } from "react"

interface ServiceCardProps {
  title: string
  description: string
  image: string
  link: string
  index: number
  isImageLoaded?: boolean
}

// Utiliser memo pour éviter les re-rendus inutiles
const ServiceCardOptimized = memo(function ServiceCardOptimized({
  title,
  description,
  image,
  link,
  index,
  isImageLoaded = false,
}: ServiceCardProps) {
  // Déterminer si cette carte doit être chargée en priorité (les 4 premières)
  const isPriority = index < 4

  return (
    <motion.div
      initial={{ opacity: 0, y: 20 }}
      whileInView={{ opacity: 1, y: 0 }}
      viewport={{ once: true }}
      transition={{ delay: index * 0.1 }}
      className="group cursor-pointer"
    >
      <Link href={link} className="block h-full">
        <div className="bg-white rounded-2xl overflow-hidden shadow-lg hover:shadow-2xl transition-all duration-500 h-full transform hover:-translate-y-2">
          <div className="relative h-64 overflow-hidden">
            <Image
              src={image || "/placeholder.svg"}
              alt={title}
              fill
              className={`object-cover transform group-hover:scale-110 transition-transform duration-500`}
              priority={isPriority}
              sizes="(max-width: 640px) 100vw, (max-width: 1024px) 50vw, 25vw"
              // Utiliser un placeholder blur simple
              placeholder="blur"
              blurDataURL="data:image/jpeg;base64,/9j/4AAQSkZJRgABAQAAAQABAAD/2wBDAAYEBQYFBAYGBQYHBwYIChAKCgkJChQODwwQFxQYGBcUFhYaHSUfGhsjHBYWICwgIyYnKSopGR8tMC0oMCUoKSj/2wBDAQcHBwoIChMKChMoGhYaKCgoKCgoKCgoKCgoKCgoKCgoKCgoKCgoKCgoKCgoKCgoKCgoKCgoKCgoKCgoKCgoKCj/wAARCAAIAAoDASIAAhEBAxEB/8QAFQABAQAAAAAAAAAAAAAAAAAAAAb/xAAhEAACAQMDBQAAAAAAAAAAAAABAgMABAUGIWGBkbHB0f/EABUBAQEAAAAAAAAAAAAAAAAAAAME/8QAGBEAAwEBAAAAAAAAAAAAAAAAAAECEZH/2gAMAwEAAhEDEQA/AJXRdBuJL22XmyjgZVDOVG+qK0aKtWfVCFm2UdSCT5qKqSRiv//Z"
            />
            <div className="absolute inset-0 bg-gradient-to-t from-black/60 to-transparent opacity-0 group-hover:opacity-100 transition-opacity duration-500" />
          </div>
          <div className="p-8">
            <h3 className="text-xl font-semibold mb-3 bg-gradient-to-r from-[#002395] to-[#001875] bg-clip-text text-transparent">
              {title}
            </h3>
            <p className="text-gray-600 mb-6 line-clamp-3">{description}</p>
            <Button
              variant="outline"
              className="w-full border-[#002395] text-[#002395] hover:bg-gradient-to-r hover:from-[#002395] hover:to-[#001875] hover:text-white transition-all duration-300"
            >
              EN SAVOIR PLUS
              <ArrowRight className="ml-2 h-4 w-4" />
            </Button>
          </div>
        </div>
      </Link>
    </motion.div>
  )
})

export default ServiceCardOptimized
