import type { ReactNode } from "react"
import Image from "next/image"
import { Breadcrumb } from "@/components/breadcrumb"
import { TableOfContents } from "@/components/table-of-contents"
import { ConsultationCTA } from "@/components/consultation-cta"
import { Card, CardContent } from "@/components/ui/card"
import Link from "next/link"

interface BreadcrumbItem {
  name: string
  url: string
}

interface TocItem {
  id: string
  title: string
}

interface RelatedService {
  title: string
  description: string
  link: string
}

interface CTAProps {
  title: string
  description: string
  buttonText: string
  buttonLink: string
}

interface ServicePageLayoutProps {
  title: string
  description: string
  breadcrumbItems: BreadcrumbItem[]
  tableOfContentsItems: TocItem[]
  headerImage: string
  showBlogWidgets?: boolean
  relatedServices?: RelatedService[]
  calculatorComponent?: ReactNode
  ctaProps?: CTAProps
  widgetOrder?: string[]
  children: ReactNode
}

export default function ServicePageLayout({
  title,
  description,
  breadcrumbItems,
  tableOfContentsItems,
  headerImage,
  showBlogWidgets = false,
  relatedServices = [],
  calculatorComponent,
  ctaProps,
  widgetOrder = ["toc", "calculator", "contact", "related"],
  children,
}: ServicePageLayoutProps) {
  // Fonction pour rendre les widgets de la sidebar dans l'ordre spécifié
  const renderSidebarWidgets = () => {
    return widgetOrder.map((widget) => {
      switch (widget) {
        case "toc":
          return (
            <div key="toc" className="mb-6">
              <TableOfContents items={tableOfContentsItems} />
            </div>
          )
        case "calculator":
          return calculatorComponent ? (
            <div key="calculator" className="mb-6">
              {calculatorComponent}
            </div>
          ) : null
        case "contact":
          return (
            <div key="contact" className="mb-6">
              <Card>
                <CardContent className="p-4">
                  <h3 className="text-lg font-semibold mb-2">Besoin d'aide ?</h3>
                  <p className="text-sm text-gray-600 mb-4">
                    Nos experts sont à votre disposition pour répondre à toutes vos questions.
                  </p>
                  <Link
                    href="/contact"
                    className="block w-full bg-[#002395] hover:bg-[#001875] text-white py-2 px-4 rounded text-center"
                  >
                    Nous contacter
                  </Link>
                </CardContent>
              </Card>
            </div>
          )
        case "related":
          return relatedServices.length > 0 ? (
            <div key="related" className="mb-6">
              <Card>
                <CardContent className="p-4">
                  <h3 className="text-lg font-semibold mb-3">Services associés</h3>
                  <div className="space-y-4">
                    {relatedServices.map((service, index) => (
                      <div key={index}>
                        <Link href={service.link} className="font-medium text-[#002395] hover:underline">
                          {service.title}
                        </Link>
                        <p className="text-sm text-gray-600 mt-1">{service.description}</p>
                      </div>
                    ))}
                  </div>
                </CardContent>
              </Card>
            </div>
          ) : null
        default:
          return null
      }
    })
  }

  return (
    <div className="min-h-screen">
      {/* En-tête avec image de fond */}
      <div className="relative h-[300px] md:h-[400px] w-full">
        <Image src={headerImage || "/placeholder.svg"} alt={title} fill className="object-cover brightness-75" />
        <div className="absolute inset-0 flex flex-col justify-center items-center text-center p-6">
          <h1 className="text-3xl md:text-4xl lg:text-5xl font-bold text-white mb-4 drop-shadow-lg">{title}</h1>
          <p className="text-lg md:text-xl text-white max-w-2xl drop-shadow-lg">{description}</p>
        </div>
      </div>

      {/* Fil d'Ariane */}
      <div className="bg-gray-100 py-3">
        <div className="container mx-auto px-4">
          <Breadcrumb items={breadcrumbItems} />
        </div>
      </div>

      {/* Contenu principal */}
      <div className="container mx-auto px-4 py-8">
        <div className="flex flex-col lg:flex-row gap-8">
          {/* Contenu principal */}
          <div className="lg:w-2/3">{children}</div>

          {/* Sidebar */}
          <div className="lg:w-1/3 space-y-6">{renderSidebarWidgets()}</div>
        </div>

        {/* CTA Section */}
        {ctaProps && (
          <div className="mt-12">
            <ConsultationCTA
              title={ctaProps.title}
              description={ctaProps.description}
              buttonText={ctaProps.buttonText}
              buttonLink={ctaProps.buttonLink}
            />
          </div>
        )}
      </div>
    </div>
  )
}
