"use client"

import { useState } from "react"
import { Share2, Bookmark, ThumbsUp } from "lucide-react"
import { cn } from "@/lib/utils"
import { useRouter } from "next/navigation"
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle } from "@/components/ui/dialog"
import { Button } from "@/components/ui/button"

interface ShareButtonsProps {
  title: string
  url: string
  className?: string
  articleId?: string // Pour identifier l'article dans la base de données
}

export function ShareButtons({ title, url, className, articleId }: ShareButtonsProps) {
  const router = useRouter()
  const [isBookmarked, setIsBookmarked] = useState(false)
  const [isLiked, setIsLiked] = useState(false)
  const [likeCount, setLikeCount] = useState(0)
  const [showAuthDialog, setShowAuthDialog] = useState(false)
  const [authAction, setAuthAction] = useState<"bookmark" | "like" | null>(null)

  // Simuler la vérification d'authentification
  const isAuthenticated = false // À remplacer par votre logique d'authentification réelle

  const handleShare = async () => {
    try {
      if (navigator.share) {
        await navigator.share({
          title: title,
          url: url,
        })
      } else {
        await navigator.clipboard.writeText(url)
        alert("Lien copié dans le presse-papier !")
      }
    } catch (error) {
      console.log("Erreur de partage:", error)
    }
  }

  const handleBookmark = () => {
    if (!isAuthenticated) {
      setAuthAction("bookmark")
      setShowAuthDialog(true)
      return
    }

    // Logique pour enregistrer le signet
    setIsBookmarked(!isBookmarked)
    // Ici, vous pourriez appeler une API pour sauvegarder le signet
  }

  const handleLike = () => {
    if (!isAuthenticated) {
      setAuthAction("like")
      setShowAuthDialog(true)
      return
    }

    // Logique pour le like
    if (!isLiked) {
      setLikeCount(likeCount + 1)
    } else {
      setLikeCount(Math.max(0, likeCount - 1))
    }
    setIsLiked(!isLiked)
    // Ici, vous pourriez appeler une API pour sauvegarder le like
  }

  const handleLogin = () => {
    router.push("/espace-client?redirect=" + encodeURIComponent(url))
    setShowAuthDialog(false)
  }

  const handleSignup = () => {
    router.push("/espace-client/inscription?redirect=" + encodeURIComponent(url))
    setShowAuthDialog(false)
  }

  return (
    <>
      <div className={cn("flex justify-between items-center py-4 border-t border-b border-gray-200", className)}>
        <div className="flex items-center gap-2">
          <span className="text-gray-700">Partager :</span>
          <button
            onClick={handleShare}
            className="p-2 hover:bg-gray-100 rounded-full transition-colors"
            aria-label="Partager cet article"
          >
            <Share2 className="h-5 w-5 text-gray-600" />
          </button>
          <button
            onClick={handleBookmark}
            className={cn("p-2 hover:bg-gray-100 rounded-full transition-colors", isBookmarked && "text-blue-600")}
            aria-label="Enregistrer cet article"
            aria-pressed={isBookmarked}
          >
            <Bookmark className={cn("h-5 w-5", isBookmarked ? "fill-blue-600 text-blue-600" : "text-gray-600")} />
          </button>
        </div>
        <button
          onClick={handleLike}
          className="flex items-center gap-2 px-3 py-1 hover:bg-gray-100 rounded-lg transition-colors"
          aria-label="Marquer comme utile"
          aria-pressed={isLiked}
        >
          <ThumbsUp className={cn("h-5 w-5", isLiked ? "fill-blue-600 text-blue-600" : "text-gray-600")} />
          <span className={cn("font-medium", isLiked ? "text-blue-600" : "text-gray-700")}>
            Utile {likeCount > 0 && `(${likeCount})`}
          </span>
        </button>
      </div>

      {/* Dialogue d'authentification */}
      <Dialog open={showAuthDialog} onOpenChange={setShowAuthDialog}>
        <DialogContent className="sm:max-w-md">
          <DialogHeader>
            <DialogTitle>Connexion requise</DialogTitle>
            <DialogDescription>
              {authAction === "bookmark"
                ? "Connectez-vous pour enregistrer cet article dans vos favoris."
                : "Connectez-vous pour marquer cet article comme utile."}
            </DialogDescription>
          </DialogHeader>
          <div className="flex flex-col gap-4 mt-4">
            <Button onClick={handleLogin} variant="default">
              Se connecter
            </Button>
            <Button onClick={handleSignup} variant="outline">
              Créer un compte
            </Button>
          </div>
        </DialogContent>
      </Dialog>
    </>
  )
}
