"use client"

import { useState } from "react"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Button } from "@/components/ui/button"
import { CheckCircle, AlertTriangle, XCircle, FileText, Code, Layers } from "lucide-react"

interface AuditItem {
  name: string
  status: "good" | "warning" | "error"
  description: string
  path?: string
}

export default function SiteAuditDashboard() {
  const [activeTab, setActiveTab] = useState("sitemap")

  // Analyse du plan du site
  const sitemapAudit: AuditItem[] = [
    // Pages principales - BONNES
    { name: "Accueil", status: "good", description: "Page d'accueil optimisée", path: "/" },
    { name: "À propos", status: "good", description: "Présentation de l'entreprise", path: "/a-propos" },
    { name: "Services", status: "good", description: "Vue d'ensemble des services", path: "/services" },
    { name: "Contact", status: "good", description: "Formulaire de contact multi-onglets", path: "/contact" },
    { name: "Tarifs", status: "good", description: "Grille tarifaire claire", path: "/tarifs" },
    { name: "FAQ", status: "good", description: "Questions fréquentes", path: "/faq" },
    { name: "Blog", status: "good", description: "Articles SEO optimisés", path: "/blog" },
    { name: "Calculateurs", status: "good", description: "Outils de calcul", path: "/calculateurs" },

    // Services spécialisés - BONS
    { name: "Gestion locative", status: "good", description: "Service principal", path: "/services/gestion-locative" },
    {
      name: "Gestion locative Paris",
      status: "good",
      description: "Service géolocalisé",
      path: "/services/gestion-locative-paris",
    },
    { name: "Transaction", status: "good", description: "Service de vente", path: "/services/transaction" },
    { name: "Juridique", status: "good", description: "Accompagnement juridique", path: "/services/juridique" },

    // Pages géographiques - BONNES mais nombreuses
    {
      name: "20 arrondissements Paris",
      status: "warning",
      description: "Pages par arrondissement - vérifier la pertinence",
      path: "/services/gestion-locative-paris/[arr]",
    },
    {
      name: "Villes Île-de-France",
      status: "warning",
      description: "Pages par ville - optimiser le contenu",
      path: "/services/gestion-locative-ile-de-france/[ville]",
    },
    {
      name: "Villes France",
      status: "warning",
      description: "Pages nationales - vérifier l'utilité",
      path: "/services/gestion-locative-france/[ville]",
    },

    // Pages administrateur de biens - REDONDANTES ?
    {
      name: "Admin biens Paris 13-14-15",
      status: "warning",
      description: "Spécialisation par arrondissements",
      path: "/services/administrateur-biens-paris-13-14-15",
    },
    {
      name: "Admin biens Paris 5-6-7",
      status: "warning",
      description: "Spécialisation par arrondissements",
      path: "/services/administrateur-biens-paris-5-6-7",
    },
    {
      name: "Admin biens Paris centre",
      status: "warning",
      description: "Spécialisation par zone",
      path: "/services/administrateur-biens-paris-centre",
    },

    // Pages potentiellement superflues
    {
      name: "Pages admin multiples",
      status: "error",
      description: "Trop de pages similaires pour admin de biens",
      path: "/services/administrateur-biens-*",
    },
    { name: "Pages test", status: "error", description: "Pages de développement à supprimer", path: "/test-*" },
    { name: "Pages redirect", status: "error", description: "Pages de redirection temporaires", path: "/redirect-*" },
  ]

  // Analyse de l'architecture des composants
  const componentAudit: AuditItem[] = [
    // Composants UI standardisés - BONS
    {
      name: "UI Components",
      status: "good",
      description: "Composants shadcn/ui standardisés",
      path: "/components/ui/*",
    },
    {
      name: "Button variants",
      status: "good",
      description: "Variantes cohérentes définies",
      path: "/components/ui/button.tsx",
    },
    {
      name: "Form components",
      status: "good",
      description: "Formulaires réutilisables",
      path: "/components/ui/input.tsx",
    },

    // Composants métier - À AMÉLIORER
    {
      name: "Service cards",
      status: "warning",
      description: "Plusieurs versions similaires",
      path: "/components/*card*.tsx",
    },
    {
      name: "Calculators",
      status: "warning",
      description: "Code dupliqué entre calculateurs",
      path: "/components/*calculator*.tsx",
    },
    {
      name: "Headers",
      status: "warning",
      description: "Multiples composants header",
      path: "/components/*header*.tsx",
    },

    // Composants spécialisés - BONS
    { name: "Blog layout", status: "good", description: "Architecture blog cohérente", path: "/components/blog-*.tsx" },
    {
      name: "SEO components",
      status: "good",
      description: "Composants SEO réutilisables",
      path: "/components/schema/*.tsx",
    },

    // Problèmes détectés
    {
      name: "Composants dupliqués",
      status: "error",
      description: "Plusieurs composants font la même chose",
      path: "Divers",
    },
    { name: "Props inconsistantes", status: "warning", description: "Interfaces non standardisées", path: "Divers" },
  ]

  // Analyse de l'architecture du code
  const codeAudit: AuditItem[] = [
    // Structure - BONNE
    { name: "App Router", status: "good", description: "Next.js 14 App Router utilisé", path: "/app/*" },
    { name: "TypeScript", status: "good", description: "Typage strict activé", path: "tsconfig.json" },
    { name: "Tailwind CSS", status: "good", description: "Styling cohérent", path: "tailwind.config.ts" },

    // Organisation - À AMÉLIORER
    { name: "Lib utilities", status: "warning", description: "Fonctions utilitaires à centraliser", path: "/lib/*" },
    { name: "Constants", status: "warning", description: "Constantes dispersées dans le code", path: "Divers" },
    { name: "Types", status: "warning", description: "Types à centraliser", path: "Divers" },

    // Performance - BONNE
    { name: "Image optimization", status: "good", description: "Next.js Image utilisé", path: "Composants" },
    { name: "Bundle size", status: "good", description: "Imports optimisés", path: "next.config.mjs" },

    // Maintenance - PROBLÈMES
    { name: "Code duplication", status: "error", description: "Logique dupliquée entre pages", path: "Divers" },
    { name: "Hardcoded values", status: "warning", description: "Valeurs en dur à externaliser", path: "Divers" },
  ]

  const getStatusIcon = (status: string) => {
    switch (status) {
      case "good":
        return <CheckCircle className="h-5 w-5 text-green-500" />
      case "warning":
        return <AlertTriangle className="h-5 w-5 text-yellow-500" />
      case "error":
        return <XCircle className="h-5 w-5 text-red-500" />
      default:
        return null
    }
  }

  const getStatusCount = (items: AuditItem[], status: string) => {
    return items.filter((item) => item.status === status).length
  }

  const renderAuditSection = (items: AuditItem[], title: string) => (
    <div className="space-y-4">
      <div className="flex gap-4 mb-6">
        <div className="flex items-center gap-2">
          <CheckCircle className="h-4 w-4 text-green-500" />
          <span className="text-sm">{getStatusCount(items, "good")} Bon</span>
        </div>
        <div className="flex items-center gap-2">
          <AlertTriangle className="h-4 w-4 text-yellow-500" />
          <span className="text-sm">{getStatusCount(items, "warning")} Attention</span>
        </div>
        <div className="flex items-center gap-2">
          <XCircle className="h-4 w-4 text-red-500" />
          <span className="text-sm">{getStatusCount(items, "error")} Problème</span>
        </div>
      </div>

      <div className="space-y-2">
        {items.map((item, index) => (
          <div key={index} className="flex items-start gap-3 p-3 border rounded-lg">
            {getStatusIcon(item.status)}
            <div className="flex-1">
              <div className="font-medium">{item.name}</div>
              <div className="text-sm text-gray-600">{item.description}</div>
              {item.path && <div className="text-xs text-gray-400 mt-1">{item.path}</div>}
            </div>
          </div>
        ))}
      </div>
    </div>
  )

  return (
    <div className="container mx-auto p-6">
      <h1 className="text-3xl font-bold mb-8">Audit Complet du Site DOGES</h1>

      <div className="flex gap-4 mb-6">
        <Button
          variant={activeTab === "sitemap" ? "default" : "outline"}
          onClick={() => setActiveTab("sitemap")}
          className="flex items-center gap-2"
        >
          <FileText className="h-4 w-4" />
          Plan du Site
        </Button>
        <Button
          variant={activeTab === "components" ? "default" : "outline"}
          onClick={() => setActiveTab("components")}
          className="flex items-center gap-2"
        >
          <Layers className="h-4 w-4" />
          Composants
        </Button>
        <Button
          variant={activeTab === "architecture" ? "default" : "outline"}
          onClick={() => setActiveTab("architecture")}
          className="flex items-center gap-2"
        >
          <Code className="h-4 w-4" />
          Architecture
        </Button>
      </div>

      <Card>
        <CardHeader>
          <CardTitle>
            {activeTab === "sitemap" && "Plan du Site - Analyse des Pages"}
            {activeTab === "components" && "Composants - Réutilisabilité"}
            {activeTab === "architecture" && "Architecture - Maintenabilité"}
          </CardTitle>
        </CardHeader>
        <CardContent>
          {activeTab === "sitemap" && renderAuditSection(sitemapAudit, "Plan du Site")}
          {activeTab === "components" && renderAuditSection(componentAudit, "Composants")}
          {activeTab === "architecture" && renderAuditSection(codeAudit, "Architecture")}
        </CardContent>
      </Card>

      {/* Recommandations */}
      <Card className="mt-8">
        <CardHeader>
          <CardTitle>🎯 Recommandations Prioritaires</CardTitle>
        </CardHeader>
        <CardContent>
          <div className="space-y-4">
            <div className="p-4 bg-red-50 border border-red-200 rounded-lg">
              <h3 className="font-semibold text-red-800 mb-2">🚨 Actions Urgentes</h3>
              <ul className="text-sm text-red-700 space-y-1">
                <li>• Supprimer les pages de test (/test-*, /admin/*, /redirect-*)</li>
                <li>• Consolider les pages administrateur de biens (trop de doublons)</li>
                <li>• Nettoyer les composants dupliqués</li>
              </ul>
            </div>

            <div className="p-4 bg-yellow-50 border border-yellow-200 rounded-lg">
              <h3 className="font-semibold text-yellow-800 mb-2">⚠️ Améliorations Recommandées</h3>
              <ul className="text-sm text-yellow-700 space-y-1">
                <li>• Standardiser les interfaces des composants</li>
                <li>• Centraliser les constantes et types</li>
                <li>• Optimiser le contenu des pages géographiques</li>
                <li>• Créer un design system documenté</li>
              </ul>
            </div>

            <div className="p-4 bg-green-50 border border-green-200 rounded-lg">
              <h3 className="font-semibold text-green-800 mb-2">✅ Points Forts</h3>
              <ul className="text-sm text-green-700 space-y-1">
                <li>• Architecture Next.js moderne et performante</li>
                <li>• Composants UI standardisés avec shadcn/ui</li>
                <li>• SEO bien optimisé avec données structurées</li>
                <li>• TypeScript et Tailwind CSS bien configurés</li>
              </ul>
            </div>
          </div>
        </CardContent>
      </Card>
    </div>
  )
}
