"use client"

import Link from "next/link"
import { usePathname } from "next/navigation"
import { Phone, User } from "lucide-react"
import { useState, useEffect } from "react"
import { Menu, X } from "lucide-react"

export default function SiteHeader() {
  const [mobileMenuOpen, setMobileMenuOpen] = useState(false)
  const [scrolled, setScrolled] = useState(false)
  const pathname = usePathname()

  // Determine if we're on the homepage
  const isHomepage = pathname === "/"

  // Handle scroll events
  useEffect(() => {
    const handleScroll = () => {
      if (window.scrollY > 50) {
        setScrolled(true)
      } else {
        setScrolled(false)
      }
    }

    window.addEventListener("scroll", handleScroll)

    // Initial check
    handleScroll()

    return () => {
      window.removeEventListener("scroll", handleScroll)
    }
  }, [])

  const isActive = (path: string) => {
    if (path === "/" && pathname !== "/") return false
    return pathname === path || pathname.startsWith(`${path}/`)
  }

  // Determine header background
  const headerBackground = scrolled
    ? "rgba(0, 35, 149, 0.95)" // Blue with opacity when scrolled (all pages)
    : isHomepage
      ? "transparent" // Transparent when at top of homepage
      : "rgba(255, 255, 255, 0.95)" // White with opacity on other pages when not scrolled

  return (
    <>
      <header
        className="fixed top-0 left-0 right-0 z-50 mt-0 pt-0 transition-all duration-300"
        style={{
          background: headerBackground,
          backdropFilter: "blur(10px)",
          WebkitBackdropFilter: "blur(10px)",
          borderBottom: "1px solid rgba(255, 255, 255, 0.1)",
          marginTop: 0,
          paddingTop: 0,
        }}
      >
        <div className="container mx-auto px-4">
          <div className="flex items-center justify-between h-24">
            <Link href="/" className="flex items-center space-x-3">
              <div className="relative w-14 h-14">
                {/* Logo background */}
                <div
                  className={`absolute inset-0 ${
                    scrolled
                      ? isHomepage
                        ? "bg-gradient-to-br from-[#002395] to-[#001875]" // Blue when scrolled on homepage
                        : "bg-gradient-to-br from-[#002395] to-[#001875]" // Blue when scrolled on other pages
                      : isHomepage
                        ? "bg-gradient-to-br from-[#002395] to-[#001875]" // Blue when not scrolled on homepage
                        : "bg-white" // White when not scrolled on other pages
                  } flex items-center justify-center font-bold text-2xl rounded-lg shadow-lg transition-all duration-300`}
                >
                  {/* Logo text */}
                  <span
                    className={
                      scrolled
                        ? isHomepage
                          ? "text-white" // White when scrolled on homepage
                          : "text-white" // White when scrolled on other pages
                        : isHomepage
                          ? "text-white" // White when not scrolled on homepage
                          : "text-[#002395]" // Blue when not scrolled on other pages
                    }
                  >
                    D
                  </span>
                </div>
              </div>
              <div className="flex flex-col">
                {/* DOGES text */}
                <span
                  className={`text-2xl font-bold ${
                    scrolled
                      ? "text-[#002395]" // Blue when scrolled (all pages)
                      : isHomepage
                        ? "bg-gradient-to-r from-[#002395] to-[#001875] bg-clip-text text-transparent" // Blue gradient when not scrolled on homepage
                        : "text-white" // White when not scrolled on other pages
                  } transition-all duration-300`}
                >
                  DOGES.
                </span>
                <span
                  className={`text-sm uppercase tracking-wider ${
                    scrolled
                      ? "text-[#002395]" // Blue when scrolled (all pages)
                      : isHomepage
                        ? "text-white" // White when not scrolled on homepage
                        : "text-white" // White when not scrolled on other pages
                  } transition-all duration-300`}
                >
                  Administrateur de biens
                </span>
              </div>
            </Link>

            {/* Navigation Links */}
            <nav className="hidden lg:flex items-center space-x-6">
              {[
                { name: "ACCUEIL", href: "/" },
                { name: "SERVICES", href: "/services" },
                { name: "TARIFS", href: "/tarifs" },
                { name: "CALCULATEURS", href: "/calculateurs" },
                { name: "FAQ", href: "/faq" },
                { name: "CONTACT", href: "/contact" },
              ].map((item) => (
                <Link
                  key={item.name}
                  href={item.href}
                  className={`text-sm font-medium transition-all duration-300 relative ${
                    scrolled
                      ? "text-[#002395] hover:text-[#002395]/80" // Blue when scrolled (all pages)
                      : isHomepage
                        ? "text-white hover:text-white/80" // White when not scrolled on homepage
                        : "text-white hover:text-white/80" // White when not scrolled on other pages
                  }`}
                >
                  {item.name}
                  <span
                    className={`absolute bottom-0 left-0 h-0.5 transition-all duration-300 w-0 hover:w-full ${
                      isActive(item.href) ? "w-full" : ""
                    } ${
                      scrolled
                        ? "bg-[#002395]" // Blue when scrolled (all pages)
                        : isHomepage
                          ? "bg-white" // White when not scrolled on homepage
                          : "bg-white" // White when not scrolled on other pages
                    }`}
                  />
                </Link>
              ))}
              {/* Appelez-nous button */}
              <a
                href="https://www.ringover.me/EU_X6dOeNDCcD+g58Q"
                target="_blank"
                rel="noopener noreferrer"
                className="inline-flex items-center justify-center px-4 py-2 rounded-md bg-gradient-to-r from-[#002395] to-[#001875] text-white text-sm font-medium shadow-lg transition-all duration-300 transform hover:scale-105"
              >
                <Phone className="h-4 w-4 mr-2" />
                <span>APPELEZ NOUS</span>
              </a>

              {/* Espace Client button */}
              <a
                href="/espace-client"
                className={`inline-flex items-center justify-center ml-2 px-4 py-2 rounded-md bg-transparent ${
                  scrolled
                    ? "border-2 border-[#002395] text-[#002395] hover:bg-[#002395] hover:text-white" // Blue border when scrolled (all pages)
                    : isHomepage
                      ? "border-2 border-white text-white hover:bg-white hover:text-[#002395]" // White border when not scrolled on homepage
                      : "border-2 border-white text-white hover:bg-white hover:text-[#002395]" // White border when not scrolled on other pages
                } text-sm font-medium transition-all duration-300 transform hover:scale-105`}
              >
                <User className="h-4 w-4 mr-2" />
                <span>ESPACE CLIENT</span>
              </a>
            </nav>

            {/* Mobile Menu Button */}
            <button
              className="lg:hidden p-2 rounded-lg hover:bg-white/10 transition-colors"
              onClick={() => setMobileMenuOpen(!mobileMenuOpen)}
              aria-label={mobileMenuOpen ? "Fermer le menu" : "Ouvrir le menu"}
            >
              <Menu
                className={
                  scrolled
                    ? "text-[#002395]" // Blue when scrolled (all pages)
                    : isHomepage
                      ? "text-white" // White when not scrolled on homepage
                      : "text-white" // White when not scrolled on other pages
                }
              />
            </button>
          </div>
        </div>
      </header>

      {/* Mobile Menu */}
      {mobileMenuOpen && (
        <div
          className="fixed top-24 left-0 right-0 z-50 lg:hidden"
          style={{
            background: "rgba(0, 35, 149, 0.95)",
            backdropFilter: "blur(10px)",
            WebkitBackdropFilter: "blur(10px)",
          }}
        >
          <div className="container mx-auto px-4 py-4">
            <div className="flex justify-end mb-4">
              <button
                className="p-2 rounded-lg hover:bg-white/10 transition-colors"
                onClick={() => setMobileMenuOpen(false)}
                aria-label="Fermer le menu"
              >
                <X className="text-white" />
              </button>
            </div>
            <nav className="flex flex-col space-y-4">
              {[
                { name: "ACCUEIL", href: "/" },
                { name: "SERVICES", href: "/services" },
                { name: "TARIFS", href: "/tarifs" },
                { name: "CALCULATEURS", href: "/calculateurs" },
                { name: "FAQ", href: "/faq" },
                { name: "CONTACT", href: "/contact" },
              ].map((item) => (
                <Link
                  key={item.name}
                  href={item.href}
                  className="text-sm font-medium text-white hover:text-white/80 transition-colors"
                  onClick={() => setMobileMenuOpen(false)}
                >
                  {item.name}
                </Link>
              ))}
              {/* Mobile Appelez-nous button */}
              <a
                href="https://www.ringover.me/EU_X6dOeNDCcD+g58Q"
                target="_blank"
                rel="noopener noreferrer"
                className="inline-flex items-center justify-center px-4 py-2 rounded-md bg-gradient-to-r from-[#002395] to-[#001875] text-white text-sm font-medium mt-2"
                onClick={() => setMobileMenuOpen(false)}
              >
                <Phone className="h-4 w-4 mr-2" />
                <span>APPELEZ NOUS</span>
              </a>

              {/* Mobile Espace Client button */}
              <a
                href="/espace-client"
                className="inline-flex items-center justify-center px-4 py-2 rounded-md bg-transparent border-2 border-white text-white text-sm font-medium mt-2 hover:bg-white hover:text-[#002395]"
                onClick={() => setMobileMenuOpen(false)}
              >
                <User className="h-4 w-4 mr-2" />
                <span>ESPACE CLIENT</span>
              </a>
            </nav>
          </div>
        </div>
      )}
    </>
  )
}
