"use client"

import type React from "react"

import { useState } from "react"
import { ArrowRight } from "lucide-react"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"

export default function SmartContactForm() {
  const [formData, setFormData] = useState({
    name: "",
    email: "",
    phone: "",
    subject: "",
    message: "",
    privacy: false,
  })

  const [isSubmitting, setIsSubmitting] = useState(false)
  const [submitResult, setSubmitResult] = useState<{ success: boolean; message: string } | null>(null)

  const handleChange = (e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement | HTMLSelectElement>) => {
    const { name, value } = e.target
    setFormData((prev) => ({ ...prev, [name]: value }))
  }

  const handleCheckboxChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const { name, checked } = e.target
    setFormData((prev) => ({ ...prev, [name]: checked }))
  }

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault()
    setIsSubmitting(true)
    setSubmitResult(null)

    try {
      // Simuler un appel API
      await new Promise((resolve) => setTimeout(resolve, 1000))

      // Simuler une réponse réussie
      setSubmitResult({
        success: true,
        message: "Votre message a été envoyé avec succès. Nous vous contacterons dans les plus brefs délais.",
      })

      // Réinitialiser le formulaire
      setFormData({
        name: "",
        email: "",
        phone: "",
        subject: "",
        message: "",
        privacy: false,
      })
    } catch (error) {
      setSubmitResult({
        success: false,
        message: "Une erreur est survenue lors de l'envoi de votre message. Veuillez réessayer.",
      })
    } finally {
      setIsSubmitting(false)
    }
  }

  return (
    <>
      {submitResult && (
        <div
          className={`mb-6 p-4 rounded-lg ${
            submitResult.success ? "bg-green-50 text-green-700" : "bg-red-50 text-red-700"
          }`}
        >
          {submitResult.message}
        </div>
      )}

      <form className="space-y-6" onSubmit={handleSubmit}>
        <div className="grid grid-cols-2 gap-6">
          <div>
            <Label htmlFor="name">Nom complet</Label>
            <Input
              id="name"
              name="name"
              type="text"
              placeholder="Votre nom"
              required
              value={formData.name}
              onChange={handleChange}
            />
          </div>
          <div>
            <Label htmlFor="email">Email</Label>
            <Input
              id="email"
              name="email"
              type="email"
              placeholder="votre@email.com"
              required
              value={formData.email}
              onChange={handleChange}
            />
          </div>
        </div>
        <div>
          <Label htmlFor="phone">Téléphone</Label>
          <Input
            id="phone"
            name="phone"
            type="tel"
            placeholder="Votre numéro de téléphone"
            value={formData.phone}
            onChange={handleChange}
          />
        </div>
        <div>
          <Label htmlFor="subject">Sujet</Label>
          <select
            id="subject"
            name="subject"
            className="w-full h-9 rounded-md border border-input bg-transparent px-3 py-1 text-sm shadow-sm transition-colors focus-visible:outline-none focus-visible:ring-1 focus-visible:ring-ring"
            required
            value={formData.subject}
            onChange={handleChange}
          >
            <option value="">Sélectionnez un sujet</option>
            <option value="gestion-locative">Gestion locative</option>
            <option value="transaction">Transaction immobilière</option>
            <option value="gli">Garantie loyers impayés</option>
            <option value="pno">Assurance PNO</option>
            <option value="syndic">Syndic de copropriété</option>
            <option value="autre">Autre demande</option>
          </select>
        </div>
        <div>
          <Label htmlFor="message">Message</Label>
          <textarea
            id="message"
            name="message"
            className="w-full min-h-[150px] rounded-md border border-input bg-transparent px-3 py-2 text-sm shadow-sm transition-colors focus-visible:outline-none focus-visible:ring-1 focus-visible:ring-ring"
            placeholder="Votre message..."
            required
            value={formData.message}
            onChange={handleChange}
          ></textarea>
        </div>
        <div className="flex items-center">
          <input
            type="checkbox"
            id="privacy"
            name="privacy"
            className="mr-2"
            required
            checked={formData.privacy}
            onChange={handleCheckboxChange}
          />
          <Label htmlFor="privacy" className="text-sm text-gray-600">
            J'accepte que mes données soient traitées conformément à la{" "}
            <a href="/politique-confidentialite" className="text-[#002395] hover:underline">
              politique de confidentialité
            </a>
          </Label>
        </div>
        <Button type="submit" variant="gradient" className="w-full text-white" disabled={isSubmitting}>
          {isSubmitting ? "Envoi en cours..." : "Envoyer le message"}
          <ArrowRight className="ml-2 h-4 w-4" />
        </Button>
      </form>
    </>
  )
}
