"use client"

import { useState } from "react"
import { Slider } from "@/components/ui/slider"
import { Button } from "@/components/ui/button"
import Link from "next/link"

export default function TransactionCalculator() {
  const [activeTab, setActiveTab] = useState<"achat" | "vente">("achat")

  // États pour l'onglet Achat
  const [propertyPrice, setPropertyPrice] = useState(350000)
  const [isNewProperty, setIsNewProperty] = useState(false)
  const [loanAmount, setLoanAmount] = useState(280000)
  const [loanPercentage, setLoanPercentage] = useState(280000 / 350000)
  const [loanDuration, setLoanDuration] = useState(20)

  // États pour l'onglet Vente
  const [sellingPrice, setSellingPrice] = useState(350000)
  const [purchasePrice, setPurchasePrice] = useState(250000)
  const [purchaseYear, setPurchaseYear] = useState(2015)
  const [remainingLoan, setRemainingLoan] = useState(100000)

  const currentYear = new Date().getFullYear()
  const ownershipYears = currentYear - purchaseYear

  // Gérer le changement du montant du prêt (onglet Achat)
  const handleLoanAmountChange = (value: number[]) => {
    const newLoanAmount = value[0]
    setLoanAmount(newLoanAmount)
    // Mettre à jour le pourcentage pour maintenir la cohérence
    setLoanPercentage(newLoanAmount / propertyPrice)
  }

  // Calculs pour l'achat
  const notaryFees = isNewProperty
    ? Math.round(propertyPrice * 0.03) // 3% pour le neuf
    : Math.round(propertyPrice * 0.08) // 8% pour l'ancien

  const agencyFees = Math.round(propertyPrice * 0.05) // 5% d'honoraires
  const totalCost = propertyPrice + notaryFees + agencyFees
  const personalContribution = totalCost - loanAmount

  // Estimation de la mensualité (très simplifiée)
  const monthlyPayment = Math.round((loanAmount * (1 + (0.035 * loanDuration) / 12)) / (loanDuration * 12))

  // Calculs pour la vente
  const sellingAgencyFees = Math.round(sellingPrice * 0.05) // 5% d'honoraires
  const capitalGain = sellingPrice - purchasePrice

  // Calcul simplifié de la taxe sur les plus-values
  let capitalGainsTax = 0
  if (capitalGain > 0 && ownershipYears < 30) {
    // Abattement pour durée de détention (simplifié)
    const reductionFactor = Math.min(1, ownershipYears / 30)
    const taxableGain = capitalGain * (1 - reductionFactor)
    capitalGainsTax = Math.round(taxableGain * 0.19) // Taux simplifié de 19%
  }

  const netSellerAmount = sellingPrice - sellingAgencyFees - remainingLoan - capitalGainsTax

  return (
    <div className="space-y-6">
      {/* Tabs */}
      <div className="grid grid-cols-2 gap-0 rounded-md overflow-hidden">
        <button
          className={`h-12 flex items-center justify-center font-medium ${
            activeTab === "achat" ? "bg-[#002395] text-white" : "bg-gray-100 text-gray-700 hover:bg-gray-200"
          }`}
          onClick={() => setActiveTab("achat")}
        >
          Achat
        </button>
        <button
          className={`h-12 flex items-center justify-center font-medium ${
            activeTab === "vente" ? "bg-[#002395] text-white" : "bg-gray-100 text-gray-700 hover:bg-gray-200"
          }`}
          onClick={() => setActiveTab("vente")}
        >
          Vente
        </button>
      </div>

      {activeTab === "achat" ? (
        // Contenu de l'onglet Achat
        <div className="grid md:grid-cols-2 gap-8">
          {/* Partie gauche - Paramètres */}
          <div>
            {/* Prix du bien */}
            <div className="mb-6">
              <div className="flex justify-between mb-2">
                <label className="font-medium">Prix du bien</label>
                <span className="font-bold">{propertyPrice.toLocaleString()} €</span>
              </div>
              <Slider
                value={[propertyPrice]}
                min={50000}
                max={1000000}
                step={10000}
                onValueChange={(value) => setPropertyPrice(value[0])}
                className="my-2"
              />
              <div className="flex justify-between text-xs text-gray-500">
                <span>50 000 €</span>
                <span>1 000 000 €</span>
              </div>
            </div>

            {/* Type de bien */}
            <div className="mb-6">
              <label className="font-medium block mb-2">Type de bien</label>
              <div className="grid grid-cols-2 gap-4">
                <button
                  className={`py-2 px-4 rounded font-medium ${
                    !isNewProperty
                      ? "bg-[#002395] text-white"
                      : "border border-gray-300 text-gray-700 hover:border-[#002395]"
                  }`}
                  onClick={() => setIsNewProperty(false)}
                >
                  Ancien
                </button>
                <button
                  className={`py-2 px-4 rounded font-medium ${
                    isNewProperty
                      ? "bg-[#002395] text-white"
                      : "border border-gray-300 text-gray-700 hover:border-[#002395]"
                  }`}
                  onClick={() => setIsNewProperty(true)}
                >
                  Neuf
                </button>
              </div>
            </div>

            {/* Montant de l'emprunt */}
            <div className="mb-6">
              <div className="flex justify-between mb-2">
                <label className="font-medium">Montant de l'emprunt</label>
                <span className="font-bold">{loanAmount.toLocaleString()} €</span>
              </div>
              <Slider
                value={[loanAmount]}
                min={0}
                max={propertyPrice}
                step={10000}
                onValueChange={handleLoanAmountChange}
                className="my-2"
              />
              <div className="flex justify-between text-xs text-gray-500">
                <span>0 €</span>
                <span>{propertyPrice.toLocaleString()} €</span>
              </div>
            </div>

            {/* Durée du prêt */}
            <div className="mb-6">
              <div className="flex justify-between mb-2">
                <label className="font-medium">Durée du prêt</label>
                <span className="font-bold">{loanDuration} ans</span>
              </div>
              <Slider
                value={[loanDuration]}
                min={5}
                max={30}
                step={1}
                onValueChange={(value) => setLoanDuration(value[0])}
                className="my-2"
              />
              <div className="flex justify-between text-xs text-gray-500">
                <span>5 ans</span>
                <span>30 ans</span>
              </div>
            </div>
          </div>

          {/* Partie droite - Résultats */}
          <div className="bg-gray-50 p-6 rounded-lg">
            <h3 className="text-xl font-bold mb-4 text-[#002395]">Résumé des coûts</h3>

            <div className="space-y-3 mb-6">
              <div className="flex justify-between">
                <span>Prix d'achat</span>
                <span className="font-bold">{propertyPrice.toLocaleString()} €</span>
              </div>
              <div className="flex justify-between">
                <span>Frais de notaire ({isNewProperty ? "3%" : "8%"})</span>
                <span className="font-bold">{notaryFees.toLocaleString()} €</span>
              </div>
              <div className="flex justify-between">
                <span>Honoraires d'agence (5%)</span>
                <span className="font-bold">{agencyFees.toLocaleString()} €</span>
              </div>
              <div className="flex justify-between pt-3 border-t border-gray-300">
                <span className="font-bold">Coût total</span>
                <span className="font-bold text-[#002395]">{totalCost.toLocaleString()} €</span>
              </div>
            </div>

            <h4 className="text-lg font-bold mb-3">Financement</h4>
            <div className="space-y-3 mb-4">
              <div className="flex justify-between">
                <span>Apport personnel</span>
                <span className="font-bold">{personalContribution.toLocaleString()} €</span>
              </div>
              <div className="flex justify-between">
                <span>Montant emprunté</span>
                <span className="font-bold">{loanAmount.toLocaleString()} €</span>
              </div>
              <div className="flex justify-between">
                <span>Mensualité estimée*</span>
                <span className="font-bold">{monthlyPayment.toLocaleString()} €</span>
              </div>
            </div>

            <p className="text-xs text-gray-500">
              *Estimation basée sur un taux d'intérêt moyen de 3,5%.
              <br />
              Contactez-nous pour une simulation personnalisée.
            </p>
          </div>
        </div>
      ) : (
        // Contenu de l'onglet Vente
        <div className="grid md:grid-cols-2 gap-8">
          {/* Partie gauche - Paramètres */}
          <div>
            {/* Prix de vente */}
            <div className="mb-6">
              <div className="flex justify-between mb-2">
                <label className="font-medium">Prix de vente estimé</label>
                <span className="font-bold">{sellingPrice.toLocaleString()} €</span>
              </div>
              <Slider
                value={[sellingPrice]}
                min={50000}
                max={1500000}
                step={10000}
                onValueChange={(value) => setSellingPrice(value[0])}
                className="my-2"
              />
              <div className="flex justify-between text-xs text-gray-500">
                <span>50 000 €</span>
                <span>1 500 000 €</span>
              </div>
            </div>

            {/* Prix d'achat initial */}
            <div className="mb-6">
              <div className="flex justify-between mb-2">
                <label className="font-medium">Prix d'achat initial</label>
                <span className="font-bold">{purchasePrice.toLocaleString()} €</span>
              </div>
              <Slider
                value={[purchasePrice]}
                min={10000}
                max={sellingPrice}
                step={10000}
                onValueChange={(value) => setPurchasePrice(value[0])}
                className="my-2"
              />
              <div className="flex justify-between text-xs text-gray-500">
                <span>10 000 €</span>
                <span>{sellingPrice.toLocaleString()} €</span>
              </div>
            </div>

            {/* Année d'achat */}
            <div className="mb-6">
              <div className="flex justify-between mb-2">
                <label className="font-medium">Année d'achat</label>
                <span className="font-bold">{purchaseYear}</span>
              </div>
              <Slider
                value={[purchaseYear]}
                min={1990}
                max={currentYear - 1}
                step={1}
                onValueChange={(value) => setPurchaseYear(value[0])}
                className="my-2"
              />
              <div className="flex justify-between text-xs text-gray-500">
                <span>1990</span>
                <span>{currentYear - 1}</span>
              </div>
            </div>

            {/* Reste à rembourser (crédit) */}
            <div className="mb-6">
              <div className="flex justify-between mb-2">
                <label className="font-medium">Reste à rembourser (crédit)</label>
                <span className="font-bold">{remainingLoan.toLocaleString()} €</span>
              </div>
              <Slider
                value={[remainingLoan]}
                min={0}
                max={purchasePrice}
                step={10000}
                onValueChange={(value) => setRemainingLoan(value[0])}
                className="my-2"
              />
              <div className="flex justify-between text-xs text-gray-500">
                <span>0 €</span>
                <span>{purchasePrice.toLocaleString()} €</span>
              </div>
            </div>
          </div>

          {/* Partie droite - Résultats */}
          <div className="bg-gray-50 p-6 rounded-lg">
            <h3 className="text-xl font-bold mb-4 text-[#002395]">Résumé de la vente</h3>

            <div className="space-y-3 mb-6">
              <div className="flex justify-between">
                <span>Prix de vente</span>
                <span className="font-bold">{sellingPrice.toLocaleString()} €</span>
              </div>
              <div className="flex justify-between">
                <span>Prix d'achat initial</span>
                <span className="font-bold">{purchasePrice.toLocaleString()} €</span>
              </div>
              <div className="flex justify-between">
                <span>Plus-value brute</span>
                <span className="font-bold">{capitalGain.toLocaleString()} €</span>
              </div>
              <div className="flex justify-between">
                <span>Durée de détention</span>
                <span className="font-bold">{ownershipYears} ans</span>
              </div>
            </div>

            <h4 className="text-lg font-bold mb-3">Frais et taxes</h4>
            <div className="space-y-3 mb-4">
              <div className="flex justify-between">
                <span>Honoraires d'agence (5%)</span>
                <span className="font-bold">{sellingAgencyFees.toLocaleString()} €</span>
              </div>
              <div className="flex justify-between">
                <span>Reste à rembourser (crédit)</span>
                <span className="font-bold">{remainingLoan.toLocaleString()} €</span>
              </div>
              {capitalGainsTax > 0 && (
                <div className="flex justify-between">
                  <span>Taxe sur plus-value*</span>
                  <span className="font-bold">{capitalGainsTax.toLocaleString()} €</span>
                </div>
              )}
              <div className="flex justify-between pt-3 border-t border-gray-300">
                <span className="font-bold">Net vendeur estimé</span>
                <span className="font-bold text-[#002395]">{netSellerAmount.toLocaleString()} €</span>
              </div>
            </div>

            <p className="text-xs text-gray-500">
              *Estimation simplifiée. La fiscalité des plus-values immobilières dépend de nombreux facteurs.
              <br />
              Contactez-nous pour une analyse personnalisée de votre situation.
            </p>
          </div>
        </div>
      )}

      <Link href="/contact">
        <Button className="w-full bg-[#002395] hover:bg-[#001875] text-white">
          Demander une estimation personnalisée
        </Button>
      </Link>
    </div>
  )
}
