"use client"

import { useEffect, useState } from "react"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Activity, Clock, Zap, BarChart3 } from "lucide-react"

interface WebVital {
  name: string
  value: number
  rating: "good" | "needs-improvement" | "poor"
  unit: string
  description: string
}

export default function WebVitalsMonitor() {
  const [vitals, setVitals] = useState<WebVital[]>([])

  useEffect(() => {
    // Fonction pour mesurer les Core Web Vitals
    const measureWebVitals = () => {
      // Simuler les mesures (en production, utiliser web-vitals library)
      const mockVitals: WebVital[] = [
        {
          name: "LCP",
          value: 2.1,
          rating: "good",
          unit: "s",
          description: "Largest Contentful Paint - Temps de chargement du plus gros élément",
        },
        {
          name: "FID",
          value: 45,
          rating: "good",
          unit: "ms",
          description: "First Input Delay - Délai avant la première interaction",
        },
        {
          name: "CLS",
          value: 0.05,
          rating: "good",
          unit: "",
          description: "Cumulative Layout Shift - Stabilité visuelle",
        },
        {
          name: "FCP",
          value: 1.2,
          rating: "good",
          unit: "s",
          description: "First Contentful Paint - Premier contenu affiché",
        },
      ]

      setVitals(mockVitals)
    }

    measureWebVitals()
  }, [])

  const getRatingColor = (rating: string) => {
    switch (rating) {
      case "good":
        return "text-green-600 bg-green-100"
      case "needs-improvement":
        return "text-yellow-600 bg-yellow-100"
      case "poor":
        return "text-red-600 bg-red-100"
      default:
        return "text-gray-600 bg-gray-100"
    }
  }

  const getIcon = (name: string) => {
    switch (name) {
      case "LCP":
        return <BarChart3 className="h-5 w-5" />
      case "FID":
        return <Zap className="h-5 w-5" />
      case "CLS":
        return <Activity className="h-5 w-5" />
      case "FCP":
        return <Clock className="h-5 w-5" />
      default:
        return <Activity className="h-5 w-5" />
    }
  }

  return (
    <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
      {vitals.map((vital, index) => (
        <Card key={index}>
          <CardHeader className="pb-2">
            <CardTitle className="flex items-center justify-between text-sm">
              <div className="flex items-center space-x-2">
                {getIcon(vital.name)}
                <span>{vital.name}</span>
              </div>
              <span className={`px-2 py-1 rounded text-xs font-medium ${getRatingColor(vital.rating)}`}>
                {vital.rating === "good" ? "Bon" : vital.rating === "needs-improvement" ? "Moyen" : "Mauvais"}
              </span>
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold mb-1">
              {vital.value}
              {vital.unit}
            </div>
            <p className="text-xs text-gray-600">{vital.description}</p>
          </CardContent>
        </Card>
      ))}
    </div>
  )
}

export { WebVitalsMonitor }
