import { blogArticles } from "./blog-utils"

/**
 * Récupère les informations d'un article à partir de son slug
 * @param slug Le slug de l'article
 * @returns Les informations de l'article ou null si l'article n'existe pas
 */
export function getArticleBySlug(slug: string) {
  return blogArticles.find((article) => article.slug === slug) || null
}

/**
 * Récupère l'image d'un article à partir de son slug
 * @param slug Le slug de l'article
 * @returns L'URL de l'image de l'article ou une image par défaut
 */
export function getArticleImage(slug: string) {
  const article = getArticleBySlug(slug)
  return article?.image || "/default-article-image.png"
}

/**
 * Récupère le titre d'un article à partir de son slug
 * @param slug Le slug de l'article
 * @returns Le titre de l'article ou une chaîne vide
 */
export function getArticleTitle(slug: string) {
  const article = getArticleBySlug(slug)
  return article?.title || ""
}

/**
 * Récupère la description d'un article à partir de son slug
 * @param slug Le slug de l'article
 * @returns La description de l'article ou une chaîne vide
 */
export function getArticleDescription(slug: string) {
  const article = getArticleBySlug(slug)
  return article?.description || ""
}

/**
 * Récupère la catégorie d'un article à partir de son slug
 * @param slug Le slug de l'article
 * @returns La catégorie de l'article ou une chaîne vide
 */
export function getArticleCategory(slug: string) {
  const article = getArticleBySlug(slug)
  return article?.category || ""
}

/**
 * Récupère la date d'un article à partir de son slug
 * @param slug Le slug de l'article
 * @returns La date de l'article ou une chaîne vide
 */
export function getArticleDate(slug: string) {
  const article = getArticleBySlug(slug)
  return article?.date || ""
}

/**
 * Récupère le temps de lecture d'un article à partir de son slug
 * @param slug Le slug de l'article
 * @returns Le temps de lecture de l'article ou une chaîne vide
 */
export function getArticleReadingTime(slug: string) {
  const article = getArticleBySlug(slug)
  return article?.readingTime || ""
}

/**
 * Récupère toutes les informations nécessaires pour un article à partir de son slug
 * @param slug Le slug de l'article
 * @returns Un objet contenant toutes les informations de l'article
 */
export function getArticleData(slug: string) {
  const article = getArticleBySlug(slug)

  return {
    title: article?.title || "Article introuvable",
    description: article?.description || "",
    image: article?.image || "/default-article-image.png",
    date: article?.date || "",
    category: article?.category || "",
    readingTime: article?.readingTime || "",
    slug: slug,
  }
}
