import { faqData, searchFAQ, type FAQItem } from "./faq-data"

export interface KnowledgeItem {
  id: string
  category: string
  keywords: string[]
  content: string
  relatedServices?: string[]
  source?: "knowledge" | "faq" | "service" | "calculator"
}

// Base de connaissances existante
export const immobilierKnowledgeBase: KnowledgeItem[] = [
  // Services principaux
  {
    id: "gestion-locative",
    category: "services",
    keywords: ["gestion locative", "gérer", "louer", "propriétaire", "bailleur", "locataire"],
    content:
      "Notre service de gestion locative comprend la recherche de locataires, l'encaissement des loyers, la gestion des travaux et le suivi administratif. Tarifs à partir de 3,9% HT des loyers encaissés.",
    relatedServices: ["garantie-loyers-impayes", "assurance-pno"],
    source: "service",
  },
  {
    id: "vente-immobiliere",
    category: "services",
    keywords: ["vendre", "vente", "estimation", "prix", "marché", "vendeur"],
    content:
      "Nous vous accompagnons dans la vente de votre bien : estimation gratuite, marketing digital, visites, négociation et signature. Commission de 3% à 5% selon le bien.",
    relatedServices: ["estimation-gratuite", "home-staging"],
    source: "service",
  },
  {
    id: "achat-immobilier",
    category: "services",
    keywords: ["acheter", "achat", "acquérir", "primo", "premier", "crédit", "financement"],
    content:
      "Nous vous aidons à trouver le bien idéal : recherche personnalisée, négociation, accompagnement crédit et signature. Pas de frais pour l'acheteur.",
    relatedServices: ["financement", "notaire"],
    source: "service",
  },
  {
    id: "investissement-locatif",
    category: "services",
    keywords: ["investissement", "investir", "rentabilité", "rendement", "défiscalisation", "LMNP", "SCI"],
    content:
      "Conseil en investissement locatif : sélection de biens rentables, optimisation fiscale, gestion complète. Rendements de 4% à 8% selon les secteurs.",
    relatedServices: ["gestion-locative", "optimisation-fiscale"],
    source: "service",
  },

  // Services spécialisés DOGES
  {
    id: "comptabilite-mandants",
    category: "services",
    keywords: ["comptabilité", "mandants", "comptable", "comptes", "bilan", "déclaration", "fiscalité"],
    content:
      "Notre service Comptabilité Mandants assure la tenue complète de vos comptes immobiliers : suivi des recettes et dépenses, établissement des décomptes annuels, déclarations fiscales, optimisation de votre fiscalité immobilière. Service dédié aux propriétaires bailleurs avec expertise spécialisée.",
    relatedServices: ["gestion-locative", "optimisation-fiscale"],
    source: "service",
  },
  {
    id: "garantie-loyers-impayes",
    category: "services",
    keywords: ["garantie", "loyers", "impayés", "assurance", "protection", "sécurité"],
    content:
      "Notre Garantie Loyers Impayés vous protège contre les risques locatifs : prise en charge des loyers impayés, frais de contentieux, dégradations. Couverture jusqu'à 100 000€ par sinistre avec franchise réduite.",
    relatedServices: ["gestion-locative", "assurance-pno"],
    source: "service",
  },
  {
    id: "assurance-pno",
    category: "services",
    keywords: ["assurance", "PNO", "propriétaire", "non", "occupant", "protection", "bien"],
    content:
      "L'Assurance Propriétaire Non Occupant (PNO) protège votre bien immobilier : dégâts des eaux, incendie, vol, responsabilité civile. Couverture adaptée aux biens locatifs avec tarifs préférentiels.",
    relatedServices: ["gestion-locative", "garantie-loyers-impayes"],
    source: "service",
  },
  {
    id: "service-juridique",
    category: "services",
    keywords: ["juridique", "droit", "contentieux", "bail", "expulsion", "procédure"],
    content:
      "Notre service juridique vous accompagne dans tous vos contentieux locatifs : rédaction de baux, procédures d'expulsion, recouvrement de créances, conseils juridiques spécialisés en droit immobilier.",
    relatedServices: ["gestion-locative", "garantie-loyers-impayes"],
    source: "service",
  },
  {
    id: "service-commercial",
    category: "services",
    keywords: ["commercial", "vente", "achat", "transaction", "négociation", "mandat"],
    content:
      "Notre équipe commerciale vous accompagne dans vos projets de transaction : estimation de biens, recherche d'acquéreurs, négociation, suivi jusqu'à la signature. Expertise du marché parisien et francilien.",
    relatedServices: ["estimation", "financement"],
    source: "service",
  },
  {
    id: "syndic-copropriete",
    category: "services",
    keywords: ["syndic", "copropriété", "assemblée", "charges", "travaux", "gestion"],
    content:
      "Notre service de syndic de copropriété assure la gestion complète de votre immeuble : assemblées générales, suivi des travaux, gestion des charges, relation avec les copropriétaires. Tarifs transparents et service de proximité.",
    relatedServices: ["gestion-immobiliere", "travaux"],
    source: "service",
  },
  {
    id: "administrateur-biens",
    category: "services",
    keywords: ["administrateur", "biens", "gestion", "patrimoine", "immobilier", "professionnel"],
    content:
      "Notre service d'administration de biens s'adresse aux professionnels et gros patrimoines : gestion globale de portefeuilles immobiliers, optimisation fiscale, reporting détaillé, conseil stratégique.",
    relatedServices: ["gestion-locative", "comptabilite-mandants"],
    source: "service",
  },
  {
    id: "gestion-professionnels",
    category: "services",
    keywords: ["professionnels", "entreprise", "bureau", "local", "commercial", "bail"],
    content:
      "Notre équipe dédiée aux professionnels vous accompagne : recherche de locaux commerciaux, négociation de baux professionnels, gestion locative spécialisée, conseil en implantation.",
    relatedServices: ["bail-commercial", "estimation"],
    source: "service",
  },

  // Secteurs géographiques
  {
    id: "paris-centre",
    category: "secteurs",
    keywords: ["paris centre", "1er", "2ème", "3ème", "4ème", "châtelet", "marais", "louvre"],
    content:
      "Paris Centre (1er-4ème) : secteur prestigieux, prix 12 000-15 000€/m², forte demande locative, rendement 3-4%. Idéal pour investissement de prestige.",
    relatedServices: ["estimation-gratuite", "gestion-locative"],
    source: "knowledge",
  },
  {
    id: "paris-ouest",
    category: "secteurs",
    keywords: ["16ème", "17ème", "neuilly", "levallois", "boulogne", "ouest parisien"],
    content:
      "Ouest parisien : secteur familial recherché, prix 8 000-12 000€/m², excellentes écoles, transports. Parfait pour familles et investissement sûr.",
    relatedServices: ["recherche-bien", "financement"],
    source: "knowledge",
  },
  {
    id: "paris-est",
    category: "secteurs",
    keywords: ["11ème", "20ème", "belleville", "république", "bastille", "est parisien"],
    content:
      "Est parisien : secteur en développement, prix 7 000-10 000€/m², quartiers dynamiques, bon potentiel de plus-value. Idéal primo-accédants.",
    relatedServices: ["achat-primo", "investissement"],
    source: "knowledge",
  },

  // Types de biens
  {
    id: "appartement-paris",
    category: "biens",
    keywords: ["appartement", "studio", "2 pièces", "3 pièces", "duplex", "loft"],
    content:
      "Appartements parisiens : du studio au 5 pièces, anciens haussmanniens ou modernes. Prix moyens : Studio 300-500k€, 2P 500-800k€, 3P 800k-1,2M€.",
    relatedServices: ["estimation-gratuite", "financement"],
    source: "knowledge",
  },
  {
    id: "bureaux-commerces",
    category: "biens",
    keywords: ["bureau", "local commercial", "commerce", "professionnel", "entreprise"],
    content:
      "Locaux professionnels : bureaux, commerces, entrepôts. Accompagnement spécialisé pour professionnels, bail commercial, optimisation fiscale.",
    relatedServices: ["gestion-professionnelle", "bail-commercial"],
    source: "knowledge",
  },

  // Financement
  {
    id: "credit-immobilier",
    category: "financement",
    keywords: ["crédit", "prêt", "financement", "banque", "taux", "apport", "mensualité"],
    content:
      "Financement immobilier : taux actuels 3,5-4,5%, apport minimum 10%, durée jusqu'à 25 ans. Nous vous accompagnons avec nos partenaires bancaires.",
    relatedServices: ["simulation-credit", "courtier"],
    source: "knowledge",
  },
  {
    id: "frais-acquisition",
    category: "financement",
    keywords: ["frais notaire", "frais acquisition", "droits mutation", "coût total"],
    content:
      "Frais d'acquisition : notaire 7-8% dans l'ancien, 2-3% dans le neuf, frais d'agence 3-5%, frais de dossier bancaire 1000€. Budget total à prévoir.",
    relatedServices: ["simulation-frais", "conseil-achat"],
    source: "knowledge",
  },

  // Investissement
  {
    id: "rentabilite-locative",
    category: "investissement",
    keywords: ["rentabilité", "rendement", "cash flow", "roi", "retour investissement"],
    content:
      "Rentabilité locative : rendement brut 3-6% à Paris, 5-8% en périphérie. Calcul : (loyer annuel / prix achat) x 100. Nous analysons chaque projet.",
    relatedServices: ["simulation-rentabilite", "conseil-investissement"],
    source: "knowledge",
  },
  {
    id: "defiscalisation",
    category: "investissement",
    keywords: ["défiscalisation", "réduction impôt", "LMNP", "Pinel", "Malraux", "fiscal"],
    content:
      "Défiscalisation immobilière : LMNP (amortissement), Pinel (réduction 12-21%), Malraux (30%), SCI. Optimisation selon votre situation fiscale.",
    relatedServices: ["conseil-fiscal", "montage-juridique"],
    source: "knowledge",
  },

  // Questions fréquentes location
  {
    id: "budget_location_conseil",
    category: "location",
    keywords: ["budget", "loyer", "charges", "revenus"],
    content:
      "Pour un budget locatif équilibré, votre loyer ne devrait pas dépasser 30% de vos revenus nets. N'oubliez pas d'inclure les charges, l'assurance habitation et les frais d'agence.",
    source: "knowledge",
  },
  {
    id: "secteurs_paris_location",
    category: "location",
    keywords: ["secteur", "arrondissement", "quartier", "transport"],
    content:
      "Les secteurs les plus demandés pour la location sont : 11e, 10e, 20e (dynamiques), 15e, 16e (familiaux), 1er-7e (prestige). Chaque secteur a ses avantages selon votre profil.",
    source: "knowledge",
  },

  // Calculateurs
  {
    id: "calculateur-credit",
    category: "calculateurs",
    keywords: ["calculateur", "crédit", "prêt", "simulation", "mensualité", "capacité", "emprunt"],
    content:
      "Notre calculateur de crédit immobilier vous permet de simuler votre capacité d'emprunt, vos mensualités et le coût total de votre crédit. Accédez-y sur notre page Calculateurs.",
    relatedServices: ["financement", "achat"],
    source: "calculator",
  },
  {
    id: "calculateur-rentabilite",
    category: "calculateurs",
    keywords: ["calculateur", "rentabilité", "investissement", "rendement", "simulation", "cash-flow"],
    content:
      "Notre calculateur de rentabilité locative vous permet d'estimer le rendement brut et net de votre investissement immobilier, ainsi que votre cash-flow mensuel. Accédez-y sur notre page Calculateurs.",
    relatedServices: ["investissement", "gestion-locative"],
    source: "calculator",
  },
  {
    id: "calculateur-frais-notaire",
    category: "calculateurs",
    keywords: ["calculateur", "frais", "notaire", "acquisition", "droits", "mutation"],
    content:
      "Notre calculateur de frais de notaire vous permet d'estimer précisément les frais d'acquisition liés à votre achat immobilier. Accédez-y sur notre page Calculateurs.",
    relatedServices: ["achat", "transaction"],
    source: "calculator",
  },
  {
    id: "calculateur-plus-value",
    category: "calculateurs",
    keywords: ["calculateur", "plus-value", "imposition", "vente", "fiscalité"],
    content:
      "Notre calculateur de plus-value immobilière vous permet d'estimer l'impôt sur la plus-value lors de la vente de votre bien. Accédez-y sur notre page Calculateurs.",
    relatedServices: ["vente", "fiscalité"],
    source: "calculator",
  },

  // Services complets DOGES
  {
    id: "tous-services-doges",
    category: "services",
    keywords: ["services", "offre", "prestations", "que faites-vous", "vos services"],
    content: `DOGES vous propose une gamme complète de services immobiliers :

🏠 **TRANSACTION**
• Vente et achat de biens immobiliers
• Estimation gratuite et expertise marché
• Accompagnement crédit et financement

🏢 **GESTION LOCATIVE**
• Gestion complète de vos biens locatifs
• Recherche et sélection de locataires
• Encaissement des loyers et suivi administratif

📊 **COMPTABILITÉ MANDANTS**
• Tenue de comptes immobiliers
• Déclarations fiscales spécialisées
• Optimisation fiscale personnalisée

🛡️ **ASSURANCES & GARANTIES**
• Garantie Loyers Impayés (GLI)
• Assurance Propriétaire Non Occupant (PNO)
• Protection juridique complète

⚖️ **SERVICES JURIDIQUES**
• Contentieux locatifs
• Rédaction de baux
• Procédures de recouvrement

🏗️ **SYNDIC DE COPROPRIÉTÉ**
• Gestion d'immeubles
• Assemblées générales
• Suivi des travaux

💼 **SERVICES PROFESSIONNELS**
• Locaux commerciaux et bureaux
• Baux professionnels
• Conseil en implantation

Contactez-nous pour un conseil personnalisé !`,
    relatedServices: ["gestion-locative", "transaction", "comptabilite-mandants"],
    source: "service",
  },
]

// Conversion des données FAQ en format KnowledgeItem
function convertFAQToKnowledge(): KnowledgeItem[] {
  return faqData.map((faq) => ({
    id: faq.id,
    category: faq.category,
    keywords: faq.keywords,
    content: `${faq.answer}`,
    source: "faq" as const,
  }))
}

// Base de connaissances complète (FAQ + connaissances)
export const completeKnowledgeBase: KnowledgeItem[] = [...immobilierKnowledgeBase, ...convertFAQToKnowledge()]

export function searchKnowledge(query: string): KnowledgeItem[] {
  const queryLower = query.toLowerCase()

  // Recherche dans la base complète (FAQ + connaissances)
  const results = completeKnowledgeBase
    .filter(
      (item) =>
        item.keywords.some((keyword) => queryLower.includes(keyword.toLowerCase())) ||
        item.content.toLowerCase().includes(queryLower),
    )
    .sort((a, b) => {
      // Prioriser les correspondances exactes dans les mots-clés
      const aExactMatch = a.keywords.some((keyword) => queryLower.includes(keyword.toLowerCase()))
      const bExactMatch = b.keywords.some((keyword) => queryLower.includes(keyword.toLowerCase()))

      if (aExactMatch && !bExactMatch) return -1
      if (!aExactMatch && bExactMatch) return 1

      // Prioriser les réponses de service qui sont plus spécifiques
      if (a.source === "service" && b.source !== "service") return -1
      if (a.source !== "service" && b.source === "service") return 1

      // Prioriser les réponses FAQ qui sont plus détaillées
      if (a.source === "faq" && b.source === "knowledge") return -1
      if (a.source === "knowledge" && b.source === "faq") return 1
      return 0
    })
    .slice(0, 3)

  return results
}

export function getKnowledgeByCategory(category: string): KnowledgeItem[] {
  return completeKnowledgeBase.filter((item) => item.category === category)
}

// Fonction spécifique pour rechercher dans la FAQ
export function searchFAQKnowledge(query: string): FAQItem[] {
  return searchFAQ(query)
}

// Statistiques de la base de connaissances
export function getKnowledgeStats() {
  const total = completeKnowledgeBase.length
  const faqItems = completeKnowledgeBase.filter((item) => item.source === "faq").length
  const knowledgeItems = completeKnowledgeBase.filter((item) => item.source === "knowledge").length
  const serviceItems = completeKnowledgeBase.filter((item) => item.source === "service").length
  const calculatorItems = completeKnowledgeBase.filter((item) => item.source === "calculator").length

  return {
    total,
    faqItems,
    knowledgeItems,
    serviceItems,
    calculatorItems,
    categories: [...new Set(completeKnowledgeBase.map((item) => item.category))],
  }
}
