export type UserProfile =
  | "acheteur_primo"
  | "vendeur"
  | "locataire"
  | "proprietaire_bailleur"
  | "investisseur"
  | "professionnel"
  | "service_inquiry"
  | "unknown"

export interface UserContext {
  messages: string[]
  detectedIntents: string[]
  budget?: string
  zone?: string
  propertyType?: string
  timeline?: string
  serviceInterest?: string
}

export class ProfileDetector {
  private context: UserContext = {
    messages: [],
    detectedIntents: [],
  }

  analyzeMessage(message: string): UserProfile {
    const messageLower = message.toLowerCase()
    this.context.messages.push(message)

    // Détecter d'abord les questions sur les services spécifiques
    if (this.isServiceInquiry(messageLower)) {
      this.context.serviceInterest = this.extractServiceType(messageLower)
      return "service_inquiry"
    }

    // Détecter les profils utilisateurs
    if (this.isAcheteur(messageLower)) {
      return "acheteur_primo"
    }

    if (this.isVendeur(messageLower)) {
      return "vendeur"
    }

    if (this.isProprietaireBailleur(messageLower)) {
      return "proprietaire_bailleur"
    }

    if (this.isLocataire(messageLower)) {
      return "locataire"
    }

    if (this.isInvestisseur(messageLower)) {
      return "investisseur"
    }

    if (this.isProfessionnel(messageLower)) {
      return "professionnel"
    }

    return "unknown"
  }

  private isServiceInquiry(message: string): boolean {
    const serviceKeywords = [
      // Administrateur de Biens
      "administrateur de biens",
      "administrateur biens",
      "administration de biens",
      "administration biens",
      "admin de biens",
      // Garantie Loyers Impayés
      "garantie loyer",
      "garanti loyer",
      "loyers impayés",
      "impayé",
      "gli",
      "protection loyer",
      // Assurance PNO
      "assurance pno",
      "propriétaire non occupant",
      "assurance propriétaire",
      // Comptabilité
      "comptabilité",
      "comptable",
      "comptes",
      "déclaration",
      "fiscalité",
      // Juridique
      "juridique",
      "contentieux",
      "expulsion",
      "procédure",
      // Syndic
      "syndic",
      "copropriété",
      "assemblée",
      "charges",
      // Services généraux
      "vos services",
      "que faites-vous",
      "prestations",
      "offre",
    ]

    return serviceKeywords.some((keyword) => message.toLowerCase().includes(keyword))
  }

  private extractServiceType(message: string): string {
    const messageLower = message.toLowerCase()

    // Vérifier d'abord Administrateur de Biens (prioritaire)
    if (
      messageLower.includes("administrateur de biens") ||
      messageLower.includes("administrateur biens") ||
      messageLower.includes("administration de biens") ||
      messageLower.includes("administration biens")
    ) {
      // Vérifier si une zone géographique est mentionnée
      if (messageLower.includes("paris")) {
        if (messageLower.includes("13") || messageLower.includes("14") || messageLower.includes("15")) {
          return "administrateur-biens-paris-13-14-15"
        } else if (messageLower.includes("5") || messageLower.includes("6") || messageLower.includes("7")) {
          return "administrateur-biens-paris-5-6-7"
        } else if (messageLower.includes("8") || messageLower.includes("16") || messageLower.includes("17")) {
          return "administrateur-biens-paris-8-16-17"
        } else if (messageLower.includes("9") || messageLower.includes("10") || messageLower.includes("18")) {
          return "administrateur-biens-paris-9-10-18"
        } else if (messageLower.includes("11") || messageLower.includes("12") || messageLower.includes("20")) {
          return "administrateur-biens-paris-11-12-20"
        } else if (messageLower.includes("19")) {
          return "administrateur-biens-paris-19"
        } else if (
          messageLower.includes("centre") ||
          messageLower.includes("1") ||
          messageLower.includes("2") ||
          messageLower.includes("3") ||
          messageLower.includes("4")
        ) {
          return "administrateur-biens-paris-centre"
        } else {
          return "administrateur-biens-paris"
        }
      } else if (messageLower.includes("hauts-de-seine") || messageLower.includes("92")) {
        return "administrateur-biens-hauts-de-seine"
      } else if (messageLower.includes("val-d'oise") || messageLower.includes("95")) {
        return "administrateur-biens-val-doise"
      } else if (messageLower.includes("haussmann") || messageLower.includes("haussmannien")) {
        return "administrateur-biens-immeubles-haussmanniens"
      } else if (messageLower.includes("ile-de-france") || messageLower.includes("idf")) {
        return "administrateur-biens-ile-de-france"
      } else {
        return "administrateur-biens-paris" // Par défaut
      }
    }

    if (messageLower.includes("garantie") || messageLower.includes("garanti") || messageLower.includes("impayé")) {
      return "garantie-loyers-impayes"
    }
    if (messageLower.includes("assurance") || messageLower.includes("pno")) {
      return "assurance-pno"
    }
    if (messageLower.includes("comptabilité") || messageLower.includes("comptable")) {
      return "comptabilite-mandants"
    }
    if (messageLower.includes("juridique") || messageLower.includes("contentieux")) {
      return "service-juridique"
    }
    if (messageLower.includes("syndic") || messageLower.includes("copropriété")) {
      return "syndic-copropriete"
    }
    return "services-generaux"
  }

  private isAcheteur(message: string): boolean {
    const acheteurKeywords = ["acheter", "achat", "acquérir", "primo", "premier achat", "devenir propriétaire"]
    return acheteurKeywords.some((keyword) => message.includes(keyword))
  }

  private isVendeur(message: string): boolean {
    const vendeurKeywords = ["vendre", "vente", "mettre en vente", "céder"]
    return vendeurKeywords.some((keyword) => message.includes(keyword))
  }

  private isProprietaireBailleur(message: string): boolean {
    const baillerKeywords = [
      "louer mon",
      "mettre en location",
      "gestion locative",
      "propriétaire bailleur",
      "gérer mon bien",
      "encaisser loyer",
    ]
    return baillerKeywords.some((keyword) => message.includes(keyword))
  }

  private isLocataire(message: string): boolean {
    const locataireKeywords = [
      "cherche un logement",
      "cherche appartement",
      "location logement",
      "louer un",
      "trouver un appartement",
      "budget mensuel",
    ]
    return locataireKeywords.some((keyword) => message.includes(keyword))
  }

  private isInvestisseur(message: string): boolean {
    const investisseurKeywords = ["investir", "investissement", "rentabilité", "rendement", "défiscalisation"]
    return investisseurKeywords.some((keyword) => message.includes(keyword))
  }

  private isProfessionnel(message: string): boolean {
    const professionnelKeywords = ["bureau", "local commercial", "entreprise", "professionnel", "commerce"]
    return professionnelKeywords.some((keyword) => message.includes(keyword))
  }

  extractBudget(message: string): string | null {
    const budgetPatterns = [/(\d+)k€?/i, /(\d+)\s*000\s*€?/i, /budget.*?(\d+)/i]

    for (const pattern of budgetPatterns) {
      const match = message.match(pattern)
      if (match) {
        return match[1] + "k€"
      }
    }
    return null
  }

  extractZone(message: string): string | null {
    const zones = [
      "paris",
      "neuilly",
      "levallois",
      "boulogne",
      "vincennes",
      "proche banlieue",
      "grande couronne",
      "ile de france",
    ]

    for (const zone of zones) {
      if (message.toLowerCase().includes(zone)) {
        return zone
      }
    }
    return null
  }

  getContext(): UserContext {
    return this.context
  }

  reset(): void {
    this.context = {
      messages: [],
      detectedIntents: [],
    }
  }
}
