const fs = require("fs")
const path = require("path")

// Fonction pour parcourir récursivement tous les fichiers
function getAllFiles(dirPath, arrayOfFiles = []) {
  const files = fs.readdirSync(dirPath)

  files.forEach((file) => {
    const fullPath = path.join(dirPath, file)

    if (fs.statSync(fullPath).isDirectory()) {
      // Ignorer certains dossiers
      if (!["node_modules", ".next", ".git", "dist", "build"].includes(file)) {
        arrayOfFiles = getAllFiles(fullPath, arrayOfFiles)
      }
    } else {
      // Inclure tous les fichiers JS/TS/JSX/TSX
      if (file.match(/\.(js|jsx|ts|tsx)$/)) {
        arrayOfFiles.push(fullPath)
      }
    }
  })

  return arrayOfFiles
}

// Fonction pour corriger les imports dans un fichier
function fixImportsInFile(filePath) {
  try {
    let content = fs.readFileSync(filePath, "utf8")
    const originalContent = content

    // Patterns à corriger
    const patterns = [
      { from: /from\s+['"]next\/Image['"]/, to: 'from "next/image"' },
      { from: /import\s+Image\s+from\s+['"]next\/Image['"]/, to: 'import Image from "next/image"' },
      { from: /import\s*{\s*Image\s*}\s*from\s+['"]next\/Image['"]/, to: 'import { Image } from "next/image"' },
      { from: /'next\/Image'/g, to: '"next/image"' },
      { from: /"next\/Image"/g, to: '"next/image"' },
    ]

    patterns.forEach((pattern) => {
      content = content.replace(pattern.from, pattern.to)
    })

    if (content !== originalContent) {
      fs.writeFileSync(filePath, content, "utf8")
      console.log(`✅ Fixed: ${filePath}`)
      return true
    }

    return false
  } catch (error) {
    console.error(`❌ Error processing ${filePath}:`, error.message)
    return false
  }
}

// Fonction principale
function main() {
  console.log("🔍 Scanning all files for incorrect Image imports...")

  const projectRoot = process.cwd()
  const allFiles = getAllFiles(projectRoot)

  console.log(`📁 Found ${allFiles.length} files to check`)

  const filesWithErrors = []
  const fixedFiles = []

  // Première passe : identifier les fichiers avec des erreurs
  allFiles.forEach((filePath) => {
    try {
      const content = fs.readFileSync(filePath, "utf8")
      if (content.includes("next/Image")) {
        filesWithErrors.push(filePath)
      }
    } catch (error) {
      // Ignorer les erreurs de lecture
    }
  })

  if (filesWithErrors.length === 0) {
    console.log("✅ No files found with incorrect Image imports!")
    return
  }

  console.log(`❌ Found ${filesWithErrors.length} files with incorrect imports:`)
  filesWithErrors.forEach((file) => {
    console.log(`  - ${file}`)
  })

  // Deuxième passe : corriger les fichiers
  console.log("\n🔧 Fixing imports...")
  filesWithErrors.forEach((filePath) => {
    if (fixImportsInFile(filePath)) {
      fixedFiles.push(filePath)
    }
  })

  console.log(`\n✅ Fixed ${fixedFiles.length} files!`)

  // Vérification finale
  console.log("\n🔍 Final verification...")
  const remainingErrors = []
  allFiles.forEach((filePath) => {
    try {
      const content = fs.readFileSync(filePath, "utf8")
      if (content.includes("next/Image")) {
        remainingErrors.push(filePath)
      }
    } catch (error) {
      // Ignorer les erreurs de lecture
    }
  })

  if (remainingErrors.length === 0) {
    console.log("✅ All imports have been successfully corrected!")
  } else {
    console.log(`❌ Still found ${remainingErrors.length} files with errors:`)
    remainingErrors.forEach((file) => {
      console.log(`  - ${file}`)
      // Afficher le contenu problématique
      try {
        const content = fs.readFileSync(file, "utf8")
        const lines = content.split("\n")
        lines.forEach((line, index) => {
          if (line.includes("next/Image")) {
            console.log(`    Line ${index + 1}: ${line.trim()}`)
          }
        })
      } catch (error) {
        console.log(`    Error reading file: ${error.message}`)
      }
    })
  }
}

if (require.main === module) {
  main()
}
