const fs = require("fs")
const path = require("path")
const { execSync } = require("child_process")

// Fonction pour parcourir récursivement tous les fichiers
function getAllFiles(dirPath, arrayOfFiles = []) {
  const files = fs.readdirSync(dirPath)

  files.forEach((file) => {
    const fullPath = path.join(dirPath, file)

    if (fs.statSync(fullPath).isDirectory()) {
      // Ignorer certains dossiers
      if (!["node_modules", ".next", ".git", "dist", "build"].includes(file)) {
        arrayOfFiles = getAllFiles(fullPath, arrayOfFiles)
      }
    } else {
      // Inclure tous les fichiers JS/TS/JSX/TSX
      if (file.match(/\.(js|jsx|ts|tsx)$/)) {
        arrayOfFiles.push(fullPath)
      }
    }
  })

  return arrayOfFiles
}

// Fonction pour corriger les imports dans un fichier
function fixImportsInFile(filePath) {
  try {
    let content = fs.readFileSync(filePath, "utf8")
    const originalContent = content

    // Patterns à corriger
    const patterns = [
      { from: /from\s+['"]next\/Image['"]/, to: 'from "next/image"' },
      { from: /import\s+Image\s+from\s+['"]next\/Image['"]/, to: 'import Image from "next/image"' },
      { from: /import\s*{\s*Image\s*}\s*from\s+['"]next\/Image['"]/, to: 'import { Image } from "next/image"' },
      { from: /'next\/Image'/g, to: '"next/image"' },
      { from: /"next\/Image"/g, to: '"next/image"' },
    ]

    patterns.forEach((pattern) => {
      content = content.replace(pattern.from, pattern.to)
    })

    if (content !== originalContent) {
      fs.writeFileSync(filePath, content, "utf8")
      console.log(`✅ Fixed: ${filePath}`)
      return true
    }

    return false
  } catch (error) {
    console.error(`❌ Error processing ${filePath}:`, error.message)
    return false
  }
}

// Créer un fichier de redirection pour next/Image
function createRedirectModule() {
  const nodeModulesPath = path.join(process.cwd(), "node_modules")
  const nextImagePath = path.join(nodeModulesPath, "next", "Image")

  // Créer le dossier si nécessaire
  if (!fs.existsSync(path.dirname(nextImagePath))) {
    fs.mkdirSync(path.dirname(nextImagePath), { recursive: true })
  }

  // Contenu du fichier de redirection
  const redirectContent = `
// Ce fichier est une redirection pour corriger les imports incorrects
// Il redirige next/Image vers next/image
module.exports = require('next/image');
`

  // Écrire le fichier
  fs.writeFileSync(nextImagePath + ".js", redirectContent)
  console.log("✅ Created redirect module at:", nextImagePath + ".js")
}

// Fonction principale
function main() {
  console.log("🚀 Starting comprehensive fix for Image imports...")

  // Approche 1: Corriger les fichiers source
  console.log("\n🔍 Approach 1: Fixing source files...")
  const projectRoot = process.cwd()
  const allFiles = getAllFiles(projectRoot)

  console.log(`📁 Found ${allFiles.length} files to check`)

  const filesWithErrors = []
  const fixedFiles = []

  // Identifier les fichiers avec des erreurs
  allFiles.forEach((filePath) => {
    try {
      const content = fs.readFileSync(filePath, "utf8")
      if (content.includes("next/Image")) {
        filesWithErrors.push(filePath)
      }
    } catch (error) {
      // Ignorer les erreurs de lecture
    }
  })

  if (filesWithErrors.length === 0) {
    console.log("✅ No files found with incorrect Image imports!")
  } else {
    console.log(`❌ Found ${filesWithErrors.length} files with incorrect imports`)

    // Corriger les fichiers
    filesWithErrors.forEach((filePath) => {
      if (fixImportsInFile(filePath)) {
        fixedFiles.push(filePath)
      }
    })

    console.log(`✅ Fixed ${fixedFiles.length} files!`)
  }

  // Approche 2: Créer un module de redirection
  console.log("\n🔍 Approach 2: Creating redirect module...")
  createRedirectModule()

  // Approche 3: Nettoyer le cache
  console.log("\n🔍 Approach 3: Cleaning build cache...")
  try {
    console.log("🧹 Removing .next directory...")
    execSync("rm -rf .next", { stdio: "inherit" })
    console.log("✅ Cache cleaned!")
  } catch (error) {
    console.log("⚠️ Error cleaning cache:", error.message)
  }

  console.log("\n✅ All approaches completed!")
  console.log("🏗️ You can now try building the project again.")
}

// Exécuter si appelé directement
if (require.main === module) {
  main()
}
